import QtQuick 2.2
import Qt.labs.platform 1.0
import "../auxiliary.js" as Code

Item {
    id: diagramItem
    objectName: "diagramItem"

    enum ActionForm
    {
        Rectangle = 0,
        Rhomb,
        Parallelogram
    }

    property bool created: false    // true after item has been dropped to field
    property bool hovered: false    // true while dragging over it
    property bool selected: false   // true when item is selected
    property point startPoint       // item coordinates before drag of whole selection
    property string actionColor:     colorAction
    property string actionTextColor: colorTextAction

    property int actionId: 0    // unique for every action
    property string actionName: "default"        // for save/load to/form file. Must be equal to filename
    property string actionTitle: qsTr("default") // for displaying in diagram
    property var parametersNames: []        // to print to log
    property var parametersVarIds: []       // 0x00000000..0xFFFFFFFF
    property var parametersVarCategories: []    // Var categories (defined in main.qml):
                                                // "const", "global", "localConst", "local" or "list"
    property ListModel parametersModel: ListModel {}
    property int permanentParametersCount: 0    // the number of permanent parameters: 0 - everyone are permanent
    property string nonPermanentParameterName   // the paramName for the added parameter
    property string nonPermanentParamType       // the paramType for the added parameter
    property string nonPermanentVarCategory     // the varCategory for the added parameter

    property string actionToolTip: qsTr("default")
    property int actionOutputCount: 1   // 2 - condition, 1 - action, 0 - finish
    property int actionFormType: Action.ActionForm.Rectangle
    property string actionType: "action"  // action, systemEvent or userEvent
    property string dialogType: "parameters"    // Type of the dialog to edit action parameters
                                                // "parameters" - ParametersDialog.qml
                                                // "script"     - ScriptDialog.qml
    property string parameterDialogSource: 'ParametersDialog.qml'   // source of the individual file of the parameter dialog

    property var outputLine: null       // not null when connected
    property var outputLineFalse: null  // not null when connected (condition only)
    property string outputLineFalsePosition // False output position: "right" or "left"

    property var inputLinesArray: []    // is not empty when line(s) are connected
    property color chainColor           // color of the chain actions borders
    property int pageIndex: 0           // page of the diagram which contains this

    property bool hasError: false    // true - if checkDiagram() have found the error

    property int itemRectMargin: Math.round(10 * kDPI)
    property real scaleFactor: 1.0

    property var buttonsItem: null  // action buttons menu

    signal itemIsAboutToDestroy(Item item)
    signal itemSelected(Item item, var mouseEvent)
    signal itemMoved(Item item, point point)
    signal itemReleased(Item item)

    width: actionOutputCount <= 1 ? diagramItemRect.width + itemRectMargin * 2
                                    : (actionFormType === Action.ActionForm.Rhomb ? rhombRect.width + itemRectMargin * 6
                                                                                 : rhombRect.width)
    height: diagramItemRect.height + itemRectMargin * 2
    z: (pageIndex === diagramContainer.pageIndex) ? zDiagramObject : zDiagramObjectOnHiddenPage
    opacity: created ? 1 : 0.5  // transparent while dragging from toolbar (creating)
    enabled: pageIndex === diagramContainer.pageIndex

    Drag.active: mouseArea.drag.active
    Drag.hotSpot.x: width / 2
    Drag.hotSpot.y: height / 2
    Drag.onDragStarted: { console.debug(objectName, "Drag.onDragStarted:") }

    transform: Scale { origin.x:0; origin.y:0; xScale:scaleFactor; yScale: scaleFactor}

    Component.onCompleted: {
        chainColor = colorChainDefault;
        diagramItem.itemSelected.connect(diagramContainer.diagramField.itemSelected)
        diagramContainer.diagramField.selectionChanged.connect(diagramItem.selectionChanged)
        diagramContainer.diagramField.selectionMoved.connect(diagramItem.selectionMoved)
        diagramContainer.diagramField.selectionReleased.connect(diagramItem.selectionReleased)
        diagramContainer.diagramField.selectionFinished.connect(diagramItem.selectionFinished)

        diagramItem.itemMoved.connect(diagramContainer.diagramField.itemMoved)
        diagramItem.itemReleased.connect(diagramContainer.diagramField.itemReleased)

        diagramContainer.diagramField.dropPosChanged.connect(diagramItem.dropPosChanged);

        diagramContainer.diagramField.dropFinished.connect(diagramItem.dropFinished);

        diagramItem.itemIsAboutToDestroy.connect(diagramContainer.diagramField.itemIsAboutToDestroy)
        console.debug(objectName, "completed")
    }
    Component.onDestruction: {
        destroyButtons();

        diagramItem.itemSelected.disconnect(diagramContainer.diagramField.itemSelected)
        diagramContainer.diagramField.selectionChanged.disconnect(diagramItem.selectionChanged)
        diagramContainer.diagramField.selectionMoved.disconnect(diagramItem.selectionMoved)
        diagramContainer.diagramField.selectionReleased.disconnect(diagramItem.selectionReleased)
        diagramContainer.diagramField.selectionFinished.disconnect(diagramItem.selectionFinished)

        diagramItem.itemMoved.disconnect(diagramContainer.diagramField.itemMoved)
        diagramItem.itemReleased.disconnect(diagramContainer.diagramField.itemReleased)

        diagramContainer.diagramField.dropPosChanged.disconnect(diagramItem.dropPosChanged);

        diagramContainer.diagramField.dropFinished.disconnect(diagramItem.dropFinished);

        diagramItem.itemIsAboutToDestroy.disconnect(diagramContainer.diagramField.itemIsAboutToDestroy)
        console.debug(objectName, "destructed")
    }

    onHeightChanged: {
        console.debug(objectName, "onHeightChanged:");
        itemReleased(diagramItem);  // to recalculate "down" line
    }

    function dump() {
        console.debug("---- action dump start",
                    objectName, actionName, pageIndex,
                    parametersVarIds, parametersVarCategories,
                    outputLine, outputLineFalse, inputLinesArray)
        var i;
        for (i = 0; i < parametersModel.count; ++i) {
            var param = parametersModel.get(i);
            console.debug("  param:", i, param.paramName, param.varName);
        }
        for (i in children)
            console.debug("  child:", i, children[i])
        console.debug("---- action dump finish")
    }

    function destroyItem() {
        itemSelected(diagramItem, undefined)
        itemIsAboutToDestroy(diagramItem);
        destroy();
        diagramContainer.diagramField.dumpChildren();
    }

    function dropPosChanged(item, dropPos) {
        if (actionType !== "action")
            return
        if (item === diagramItem)
            return;

        console.debug("dropPosChanged()", objectName, item.objectName, dropPos)

        var pointInThis = mapFromItem(diagramContainer.diagramField, dropPos.x, dropPos.y)
        var pointInDiagramItemRect = mapToItem(diagramItemRect, pointInThis.x, pointInThis.y)
        var diagramItemRectHovered = diagramItemRect.contains(Qt.point(pointInDiagramItemRect.x, pointInDiagramItemRect.y))
        diagramItem.hovered = diagramItemRectHovered
    }

    function dropFinished() {
        if (diagramItem.hovered) {

        }
        diagramItem.hovered = false
    }

    function selectionChanged(item, mouseEvent, itemWasSelected) {
        console.debug(objectName, "selectionChanged()", item)

        /*if (pageIndex !== diagramContainer.pageIndex)
            return;*/

        //! select self
        if (item === diagramItem) {
            if ((mouseEvent !== undefined)
                    && (mouseEvent.modifiers & Qt.ControlModifier))
            {
                diagramItem.selected = !diagramItem.selected
            }
            else
            {
                diagramItem.selected = true;
            }
            startPoint = Qt.point(diagramItem.x, diagramItem.y);

            if (diagramItem.selected === true)
                showButtonsDelayed();
            else
                hideButtonsDelayed();

            return;
        }
        else {
            hideButtonsDelayed();
        }

        //! select already selected item
        if (itemWasSelected)
            return;

        //! select new item (previously not selected)
        // don't release selection in case of Ctrl is pressed
        if ((mouseEvent !== undefined)
                && (mouseEvent.modifiers & Qt.ControlModifier))
            return;

        // deselect self at last
        diagramItem.selected = false;
    }

    function selectionMoved(item, deltaX, deltaY) {
        console.debug(objectName, "selectionMoved()", item, deltaX, deltaY)

        if (pageIndex !== diagramContainer.pageIndex)
            return;

        // ignore self
        if (item === diagramItem)
            return;
        if (diagramItem.selected === false)
            return;

        x = startPoint.x + deltaX;
        y = startPoint.y + deltaY;
        diagramContainer.diagramField.alignItem(diagramItem);
    }

    function selectionReleased(item) {
        console.debug(objectName, "selectionReleased()", item)

        if (pageIndex !== diagramContainer.pageIndex)
            return;

        if (diagramItem.selected)
            startPoint = Qt.point(diagramItem.x, diagramItem.y);
    }

    //! Select self if it is inside selectedRect
    function selectionFinished(selectedRect) {
        console.debug(objectName, "selectionFinished()", selectedRect);

        if (pageIndex !== diagramContainer.pageIndex)
            return;

        if ((selectedRect.x <= x)
                && (selectedRect.y <= y)
                && ((x + width) <= (selectedRect.x + selectedRect.width))
                && ((y + height) <= (selectedRect.y + selectedRect.height)))
        {
            selectionChanged(diagramItem, undefined, true);
        }
    }

    function showParametersDialog() {
        hasError = false;
        if (dialogType === "parameters")
            diagramContainer.loadAndCreateParametersWindow(diagramItem);
        else if (dialogType === "script")
            diagramContainer.showScriptDialog(diagramItem);
        else if(dialogType === "can_parser")
            diagramContainer.showCanParserDialog(diagramItem);
    }

    function showColorDialog() {
        colorDialog.open();
    }

    //! Update properties which is needed for saving
    function prepareForSaving() {
        console.debug(actionName, arguments.callee.name)
        parametersNames.length = 0;
        parametersVarIds.length = 0;
        parametersVarCategories.length = 0;
        // process every parameter
        var j, k
        for (j = 0; j < parametersModel.count; ++j) {
            var param = parametersModel.get(j)

            console.debug(" ", actionName, j, param.paramName, param.varName)

            parametersNames[j] = param.paramName

            // search index for list param
            if (param.paramType === categoryList) {
                for (k = 0; k < param.paramList.count; ++k)
                    if (param.paramList.get(k).varName === param.varName) {
                        parametersVarIds[j] = k;
                        parametersVarCategories[j] = categoryList;
                        break;
                    }
                if (k !== param.paramList.count) // varName was found
                    continue;
            }

            // Рассмотрим категорию canIndex
            if (param.paramType === categoryCanIndex) {
                parametersVarIds[j] = param.varName;
                parametersVarCategories[j] = categoryCanIndex;
                continue;
            }

            // search in const vars
            var id = diagramContainer.diagramField.getVarIdFromProxyModel(constVarsProxyModel, param)
            if (id !== -1)
            {
                parametersVarIds[j] = id;
                parametersVarCategories[j] = categoryConst;
                continue;
            }

            // search in global vars
            id = diagramContainer.diagramField.getVarIdFromProxyModel(globalVarsProxyModel, param)
            if (id !== -1)
            {
                parametersVarIds[j] = id;
                parametersVarCategories[j] = categoryGlobal;
                continue;
            }

            // search in localConst vars
            id = diagramContainer.diagramField.getVarIdFromProxyModel(localConstVarsProxyModel, param)
            if (id !== -1)
            {
                parametersVarIds[j] = id;
                parametersVarCategories[j] = categoryLocalConst;
                continue;
            }

            // search in local vars
            id = diagramContainer.diagramField.getVarIdFromProxyModel(localVarsProxyModel, param)
            if (id !== -1)
            {
                parametersVarIds[j] = id;
                parametersVarCategories[j] = categoryLocal;
                continue;
            }

            // varName not found
            parametersVarIds[j] = ""
            parametersVarCategories[j] = ""
        }
        console.debug(actionName, arguments.callee.name,
                      parametersNames, parametersVarIds,
                      parametersVarCategories)
    }

    // create buttons on enter
    function showButtonsDelayed() {
        if (buttonsItem === null)
            buttonsTimer.start();
        else
            buttonsItem.destroyTimer.stop();
    }

    // destroy buttons on exit
    function hideButtonsDelayed() {
        if (buttonsItem === null) {
            buttonsTimer.stop();
        }
        else {
            buttonsItem.destroyTimer.start();
        }
    }

    function createButtons () {
        if (!mouseArea.containsMouse)
            return;

        buttonsItem = Code.loadAndCreate("actions/ActionButtons.qml",
                                         diagramContainer.diagramField);
        buttonsItem.actionItem = diagramItem;
        buttonsItem.anchors.left = diagramItem.right;
        buttonsItem.anchors.verticalCenter = diagramItem.verticalCenter;
        buttonsItem.z = Qt.binding( function() {
            return (pageIndex === diagramContainer.pageIndex)
                    ? zVarPanel : zDiagramObjectOnHiddenPage
        })  // to hide when switch to another page
    }

    function destroyButtons() {
        buttonsTimer.stop();
        if (buttonsItem !== null)
            buttonsItem.destroyItem();
    }

    Timer {
        id: buttonsTimer

        interval: buttonsTimerInterval
        onTriggered: createButtons()
    }

    Rectangle {
        id: diagramItemVisualizer
        objectName: "diagramItemVisualizer"
        anchors.centerIn: parent
        width: actionFormType === Action.ActionForm.Parallelogram ? rhombRect.width + itemRectMargin * 4 : parent.width
        height:  parent.height
        radius: Math.round(6 * kDPI)
        color: "orangered"
        opacity: 0.5
        visible: diagramItem.hasError
    }

    Rectangle {
        id: rhombRect
        objectName: "rhombRect"

        property int sideLength: diagramItemRect.width / Math.SQRT2 + radius * 4
        property bool isRhomb: actionFormType === Action.ActionForm.Rhomb

        anchors.centerIn: diagramItemRect
        width:  isRhomb ? sideLength : 1.3 * sideLength
        height: isRhomb ? sideLength : Math.max(diagramContainer.diagramField.itemHeight + radius,
                                                            diagramItemRectText.paintedHeight * 1.2 + diagramItemRectText.anchors.margins * 4)
        radius: imageHeight
        visible: (actionType === "action") && (actionOutputCount == 2) && (actionFormType !== Action.ActionForm.Rectangle)  // when it is Condition action or Event action
        color: actionColor
        border.width: isRhomb ? 2 * regularThickness : 1.2 * regularThickness
        border.color: chainColor

        transform: [
            Rotation {
                origin.x: rhombRect.width / 2
                origin.y: rhombRect.height / 2
                angle: 45
            },
            Scale {
                id: scale
                origin.x: rhombRect.width / 2
                origin.y: rhombRect.height / 2
                xScale: 1.0
                yScale: rhombRect.isRhomb ? (diagramItemRect.height + 10) / (rhombRect.height * Math.SQRT2) : 0.7
            },
            Rotation {
                origin.x: rhombRect.width / 2
                origin.y: rhombRect.height / 2
                angle: rhombRect.isRhomb ? 0 : -50 * scale.yScale
            }
        ]

    }

    Rectangle {
        id: diagramItemRect
        objectName: "diagramItemRect"
        visible: (actionType !== "action") || (actionOutputCount < 2)   // is not Condition action
        width: ((actionType === "action") && (actionOutputCount == 2)) ? diagramItemRectText.paintedWidth + 20 //размер для блока условия
                                                                       : Math.max(diagramContainer.diagramField.itemWidth,
                        (Math.round((diagramItemRectText.paintedWidth + diagramItemRectText.anchors.margins*2)
                                    / (diagramContainer.diagramField.objectsGridSize2x*2))+1)
                        * (diagramContainer.diagramField.objectsGridSize2x*2))
        height: Math.max(diagramContainer.diagramField.itemHeight,
                         diagramItemRectText.paintedHeight + diagramItemRectText.anchors.margins*2)
        radius: ((actionType !== "action") || (actionId == 0x00))
                ? (height / Math.round(2 * kDPI))
                : Math.round(12 * kDPI)
        anchors.centerIn: parent
        color: actionColor
        border.width: regularThickness
        border.color: chainColor
    }

    Text {
        id: diagramItemRectText
        objectName: "diagramItemRectText"
        anchors.fill: diagramItemRect
        anchors.margins: parameterItemMargins
        anchors.bottomMargin: (actionFormType === Action.ActionForm.Parallelogram) ? parameterItemMargins / 2 : parameterItemMargins
        wrapMode: Text.WordWrap
        verticalAlignment: (contentHeight > height) ? Text.AlignTop : Text.AlignVCenter
        horizontalAlignment: Text.AlignHCenter
        color: actionTextColor
        font.family: "Verdana"
        font.pointSize: 8
        font.letterSpacing: 0.5
        text: diagramItem.actionTitle
        clip: true
    }

    MouseArea {
        id: mouseArea
        objectName: "mouseArea"

        anchors.fill: parent

        drag.target: diagramItem
        drag.minimumX: -(diagramItem.width - diagramItemRect.width) / 2
        drag.minimumY: -(diagramItem.height - diagramItemRect.height) / 2
        drag.maximumX: diagramContainer.diagramField.width - diagramItem.width + (diagramItem.width - diagramItemRect.width) / 2
        drag.maximumY: diagramContainer.diagramField.height - diagramItem.height + (diagramItem.height - diagramItemRect.height) / 2

        hoverEnabled: true

        onEntered: showButtonsDelayed();
        onExited: hideButtonsDelayed();

        onPressed: {
            console.debug('mouseArea.onPressed:', diagramItem.x, diagramItem.y,
                          mouseX, mouseY)
            itemSelected(diagramItem, mouse)
        }
        onPositionChanged: {
            if (mouseArea.pressed) {
                itemMoved(diagramItem, Qt.point(mouseX, mouseY))
                diagramContainer.diagramField.alignItem(diagramItem)
            }
        }
        onReleased: {
            itemReleased(diagramItem);
        }
        onDoubleClicked: {
            if (parametersModel.count)
                showParametersDialog();
        }

        DragPoint {
            id: dragPointTrue
            objectName: "dragPointTrue"

            anchors.bottom: parent.bottom
            anchors.bottomMargin: itemRectMargin - radius
            anchors.horizontalCenter: parent.horizontalCenter

            dragItem: diagramItem
            visible: actionOutputCount
            textVisible: (actionType === "action") && (actionOutputCount == 2)   // only when it is Condition action
            pointType: pointTypeTrue
            outputLine: diagramItem.outputLine
            decoratorVisible: actionOutputCount && (mouseArea.containsMouse && !mouseArea.drag.active)
            showImage: actionOutputCount

            Component.onCompleted: {
                dragPosChanged.connect(diagramContainer.diagramField.dragPosChanged)
            }
            Component.onDestruction: {
                dragPosChanged.disconnect(diagramContainer.diagramField.dragPosChanged)
            }
        }
        DragPoint {
            id: dragPointFalseRight
            objectName: "dragPointFalseRight"

            anchors.right: parent.right
            anchors.rightMargin: itemRectMargin - radius
            anchors.verticalCenter: parent.verticalCenter

            dragItem: diagramItem
            visible: (2 <= actionOutputCount)
            textVisible: (actionType === "action") && (actionOutputCount == 2)   // only when it is Condition action
            pointType: pointTypeFalse
            pointPosition: "right"
            outputLine: diagramItem.outputLineFalse
            outputLinePosition: diagramItem.outputLineFalsePosition
            decoratorVisible: (2 <= actionOutputCount)
                              && (mouseArea.containsMouse && !mouseArea.drag.active)
            showImage: (2 <= actionOutputCount)

            Component.onCompleted: {
                dragPosChanged.connect(diagramContainer.diagramField.dragPosChanged)
            }
            Component.onDestruction: {
                dragPosChanged.disconnect(diagramContainer.diagramField.dragPosChanged)
            }
        }
        DragPoint {
            id: dragPointFalseLeft
            objectName: "dragPointFalseLeft"

            anchors.left: parent.left
            anchors.leftMargin: itemRectMargin - radius
            anchors.verticalCenter: parent.verticalCenter

            dragItem: diagramItem
            visible: (2 <= actionOutputCount)
            textVisible: (actionType === "action") && (actionOutputCount == 2)   // only when it is Condition action
            pointType: pointTypeFalse
            pointPosition: "left"
            outputLine: diagramItem.outputLineFalse
            outputLinePosition: diagramItem.outputLineFalsePosition
            decoratorVisible: (2 <= actionOutputCount)
                              && (mouseArea.containsMouse && !mouseArea.drag.active)
            showImage: (2 <= actionOutputCount)

            Component.onCompleted: {
                dragPosChanged.connect(diagramContainer.diagramField.dragPosChanged)
            }
            Component.onDestruction: {
                dragPosChanged.disconnect(diagramContainer.diagramField.dragPosChanged)
            }
        }
    }

    ColorDialog {
        id: colorDialog
        objectName: "colorDialog"
        modality: Qt.ApplicationModal
        title: qsTr("Choose color for the chain")

        onAccepted: {
            console.log("Color:", colorDialog.color,
                        "SelectedColor:", colorDialog.currentColor);
            diagramContainer.diagramField.setChainColor(diagramItem, colorDialog.currentColor);
        }
        flags: dialogWindowFlags
    }

    states: [
        State {
            name: "dragActive"
            extend: "selected"
            when: mouseArea.drag.active
            PropertyChanges { target: diagramItem; opacity: 0.5 }
            PropertyChanges { target: diagramItem; z: zDiagramDraggingObject }
        },
        State {
            name: "dropReady"
            when: hovered
            PropertyChanges { target: diagramItemRect; border.width: selectedThickness }
            PropertyChanges { target: rhombRect; border.width: actionFormType === Action.ActionForm.Parallelogram ? 1.2 * selectedThickness : selectedThickness * 2 }
        },
        State {
            name: "selected"
            when: selected
            PropertyChanges { target: diagramItemRect; border.color: colorSelected }
            PropertyChanges { target: diagramItemRect; border.width: selectedThickness }
            PropertyChanges { target: rhombRect; border.color: colorSelected }
            PropertyChanges { target: rhombRect; border.width: actionFormType === Action.ActionForm.Parallelogram ? 1.2 * selectedThickness : selectedThickness * 2}
        }
    ]
}
