import QtQuick 2.2
import QtQuick.Controls 1.2
import Qt.labs.platform 1.0
import com.galileosky.cppclasses 1.0
import "actions/view_types.js" as Vtypes // типы фильтров. Глобальные переменные.

// Класс отвечающий за блоки на поле

DropArea {
    id: diagramField
    objectName: "diagramField"

    property int draggedItemCount: 0    //!< Номер перетаскиваемого элемента
                                        //!< Вставляется в имя создаваемого объекта, обеспечивая уникальность имени
    property Item dragTarget: null  // item being dragged
    property string dragTargetType: ""  // type of dragTarget to connect with dropTarget
    property Item dropTarget: null  // item to drop
    property Component dynamicComponent: null
    property Item dynamicItem: null

    property int linesGridSize: Math.round(8 * kDPI)
    property int objectsGridSize2x: linesGridSize * 2
    property int itemWidth:  linesGridSize * 20
    property int itemHeight: linesGridSize * 8

    signal dropPosChanged(Item item, point dropPoint)
    signal dropFinished()
    signal selectionChanged(Item item, var mouseEvent, bool itemWasSelected)
    signal selectionMoved(Item item, var deltaX, var deltaY)
    signal selectionReleased(Item item)
    signal selectionFinished(rect selectRect)
    property real diagramCenterPointX: 0.0
    property real diagramCenterPointY: 0.0
    property real diagramScaleFactor: 1.0

    property int last_mapped_pos_x: 0
    property int last_mapped_pos_y: 0

    property bool is_filter: false

    function numAscendingOrder(a, b) { return (a-b); }
    function numDescendingOrder(a, b) { return (b-a); }
    function decrement(element, index, array) {
        array[index] = element - 1;
    }
    function unique(value, index, self) {
        return self.indexOf(value) === index;
    }

    function findIndex(list, item) {
        if (item === undefined)
            return -1;

        var i, object;
        for (i = 0; i < list.length; ++i) {
            object = list[i]
            if (object === item) {
                return i;
            }
        }
        return -1;
    }

    //! Search and return diagramObject by \a inputLineIndex
    function findItemByInputLine(line) {
        if (line === null)
            return undefined;

        for (var i = 0 ; i < diagramField.children.length; ++i) {
            // process only diagramObject_X
            var object = diagramField.children[i]
            if (object.visible === false)
                continue;
            if (object.objectName.indexOf("diagramObject_") === -1)
                continue;
            if (object.created === false)
                continue;

            if (object.inputLinesArray.indexOf(line) !== -1)
                return object;
        }
        return undefined;
    }
    //! Search and return diagramObject by one of the outputLines
    function findItemByOutputLine(line)
    {
        if (line === null)
            return undefined;

        for (var i = 0 ; i < diagramField.children.length; ++i) {
            // process only diagramObject_X
            var object = diagramField.children[i]
            if (object.visible === false)
                continue;
            if (object.objectName.indexOf("diagramObject_") === -1)
                continue;
            if (object.created === false)
                continue;

            if (object.outputLine === line)
                return object;
            if (object.outputLineFalse === line)
                return object;
        }
        return undefined;
    }

    //! Проверяем линию на точ то она имеет элементы с обоих сторон
    function checkLine(line) {
        return findItemByOutputLine(line)!== undefined && findItemByInputLine(line)!== undefined;
    }


    function dumpChildren()
    {
        console.debug("---- Dump the list of diagramField.children ----")
        for (var i = 0; i<diagramField.children.length; i += 1)
        {
            console.debug(i, diagramField.children[i].visible,
                        diagramField.children[i].objectName,
                        diagramField.children[i].pageIndex,
                        diagramField.children[i].x,
                        diagramField.children[i].y,
                        diagramField.children[i].z,
                        diagramField.children[i].width,
                        diagramField.children[i].height,
                        diagramField.children[i].outputLine,
                        diagramField.children[i].outputLineFalse,
                        diagramField.children[i].inputLinesArray,
                        diagramField.children[i].actionName,
                        diagramField.children[i].parametersVarIds,
                        diagramField.children[i].parametersVarCategories
                        );
        }
        for (i = 0; i < diagramField.children.length; i += 1)
        {
            console.debug(i,
                        diagramField.children[i].objectName,
                        "\nOut Line:       ",
                        diagramField.children[i].outputLine,
                        "\nOut Line False: ",
                        diagramField.children[i].outputLineFalse,
                        "\ninput Lines:    ",
                        diagramField.children[i].inputLinesArray
                        );
        }
        console.debug("------------------------------------------------")
    }

    function loadAndCreateItem(source, posnInWindow)
    {
        console.debug(arguments.callee.name, source, posnInWindow);

        dynamicComponent = Qt.createComponent(source);
        console.debug("dynamicComponent.status:",
                      dynamicComponent.status, dynamicComponent.errorString())

        dynamicItem = dynamicComponent.createObject(diagramField)
        if (dynamicComponent.dynamicItem === null) {
            console.debug("createItem(): error");
            return;
        }

        console.debug("createItem(): created draggedItem = " + dynamicItem);
        if (dynamicItem.endpoint1) {
            dynamicItem.endpoint1.x = posnInWindow.x;
            dynamicItem.endpoint1.y = posnInWindow.y;
            dynamicItem.objectName = 'diagramLine_' + draggedItemCount;
            dynamicItem.is_filter = is_filter;
        }
        // creating objects after loading diagram from file
        else {
            dynamicItem.x = posnInWindow.x;
            dynamicItem.y = posnInWindow.y;
            dynamicItem.objectName = 'diagramObject_' + draggedItemCount;
            dynamicItem.startPoint = posnInWindow;
            draggedItemCount += 1;
            dynamicItem = null;
        }
    }

    // restore parameters of last created diagramObject_ from its parameter lists
    function restoreParameters() {
        console.debug("---- restoreParameters: start");

        var i, item = diagramField.children[diagramField.children.length-1];
        console.debug(item.objectName);
        // вывод идентификаторов переменных
        for (i = 0; i < item.parametersVarIds.length; ++i){
            console.log("pid", i, ": ", item.parametersVarIds[i]);
        }

        for (i = 0; i < globalVarsProxyModel.count; ++i){
            console.log("global vars", i, ": ", globalVarsProxyModel.get(i, "varName"));
        }

        for (i = 0; i < item.parametersVarIds.length; ++i)
        {
            var pId = item.parametersVarIds[i];
            var varName;
            var varCategory = item.parametersVarCategories[i]

            if (varCategory === categoryList)
            {
                for (var j = 0; j < item.parametersModel.get(i).paramList.count; j++) {
                    if (item.parametersModel.get(i).paramList.get(j).value === pId) {
                        pId = j;
                        break;
                    }
                }

                item.parametersModel.set(i, {
                        "varName":
                        item.parametersModel.get(i).paramList.get(pId).varName,
                        "varBaseCategory":
                        item.parametersModel.get(i).paramList.get(pId).varBaseCategory});
            }
            else if (varCategory === categoryCanIndex)
            {
                // Поскольку мы создали отдельную категорию переменных индексов битов для CAN.
                // Мы считаем что она у нас встречается только в блоке работы с CAN.
                // В силу особенностей восстановления переменных из Json переменнные хранящие знаячение
                // с CAN шины нужно добавлять отдельно, а не как обычно.
                // Поэтому если мы нашли индекс для store begin #, то и предыдущий эдемент а именно имя переменной зададим здесь же.

                // Но сначала нужно восстановить индексы как в списке.
                // 1 определим первый или второй индекс перед нами.
                // Количество переменных в группе, у нас их 4
                var amount_of_group_vars = 4;
                var pos_offset = 3;

                if (0 === (i-pos_offset) % amount_of_group_vars) //begin_index
                {
                    var end_index = (i-1 -2)/amount_of_group_vars + 1;
                    item.parametersModel.set(i, {
                                                 "paramName": "store end %1".arg(end_index),
                                                 "paramType": categoryCanIndex,
                                                 "varCategory": categoryCanIndex,
                                                 "varName": "%1".arg(pId), // Id из json это и есть имя переменной, это индекс.
                                                 "varBaseCategory":categoryCanIndex,
                                                 "isEditable": false,
                                             });
                    // здесь же проверим предыдущий аргумент. проверяем создан ли он или нет.
                    if (undefined === item.parametersModel.get(i-1).paramName)
                    {
                        // Сзааем переменную
                        item.parametersModel.set(i-1,
                                                 {
                                                   "paramName": "store var %1".arg(end_index),
                                                   "paramType": "int",
                                                   "varCategory": "(global)",
                                                   "varBaseCategory":categoryGlobal,
                                                   "isEditable" : true,
                                                   "isVisible" : true,
                                                 })
                    }
                }
                pos_offset++;
                if (0 === (i-pos_offset) % amount_of_group_vars)
                {
                    var begin_index = (i-2 -2)/amount_of_group_vars + 1;
                    item.parametersModel.set(i, {
                                                 "paramName": "store begin %1".arg(begin_index),
                                                 "paramType": categoryCanIndex,
                                                 "varCategory": categoryCanIndex,
                                                 "varName": "%1".arg(pId), // Id из json это и есть имя переменной, это индекс.
                                                 "varBaseCategory":categoryCanIndex,
                                                 "isEditable": false,
                                                });
                }
                pos_offset++;
                if (0 === (i-pos_offset) % amount_of_group_vars)
                {
                    var endian_index = (i-3 -2)/amount_of_group_vars + 1;
                    item.parametersModel.set(i, {
                                                 "paramName":  "store endian %1".arg(begin_index),
                                                 "paramType": categoryCanIndex,
                                                 "varCategory": categoryCanIndex,
                                                 "varName": "%1".arg(pId), // Id из json это и есть имя переменной, это индекс.
                                                 "varBaseCategory":categoryCanIndex,
                                                 "isEditable": false,
                                                });
                }
            }
            else {
                if (pId === -1)
                {
                    varName = ""
                }
                else
                {
                    if (varCategory === categoryConst)
                    {
                        varName = constVarsProxyModel.get(pId, "varName");
                    }
                    else if (varCategory === categoryGlobal)
                    {
                        varName = globalVarsProxyModel.get(pId, "varName");
                    }
                    else if (varCategory === categoryLocalConst)
                    {
                        varName = localConstVarsProxyModel.get(pId, "varName");
                    }
                    else if (varCategory === categoryLocal)
                    {
                        varName = localVarsProxyModel.get(pId, "varName");
                    }
                }
                item.parametersModel.set(i, { "varName":varName, "varBaseCategory":varCategory});
            }

            // restoring non-permanent parameters
            if ((item.permanentParametersCount > 0) && (item.permanentParametersCount <= i))
            {
                item.parametersModel.set(i, {
                        "paramName":item.nonPermanentParameterName,
                        "paramType":item.nonPermanentParamType,
                        "varCategory":item.nonPermanentVarCategory,
                        "varName":varName,
                        "varBaseCategory":varCategory,
                        "isEditable":true,
                        "isVisible": true});
            }
        }

        if (item.actionName === "TarFilter" || item.actionName === "FuelTarFilter") // Исправление при загрузке параметров из таблицы
        {
            var index = item.actionName === "TarFilter" ? 2 : 5;
            var lastIndex = item.parametersModel.count - 1;
            var tempName = item.parametersModel.get(index).paramName;
            var tempType = item.parametersModel.get(index).paramType;

            item.parametersModel.set(index, {"paramName":item.parametersModel.get(lastIndex).paramName , "paramType":item.parametersModel.get(lastIndex).paramType});
            item.parametersModel.set(lastIndex, {"paramName":tempName, "paramType":tempType});

            for (var l = index; l < lastIndex; ++l)
                item.parametersModel.set(l, {"paramType":item.nonPermanentParamType, "varCategory":item.nonPermanentVarCategory});
        }

        item.dump();
        console.debug("---- restoreParameters: finish");
    }

    function restoreConnections()
    {
        console.debug("---- restoreConnections: start");
        dumpChildren();

        var connections = []; // [outputObjectIndex, outputLineType, inputObjectIndex, pageIndex]

        var i,j;
        var object;

        // push outputs to connections[]
        for (i = 0; i < diagramField.children.length; ++i)
        {
            // omit object expcept diagramObject_N
            object = diagramField.children[i]

            if (object.visible === false)
                continue;
            if (object.objectName.indexOf("diagramObject_") !== 0)
                continue;
            if (object.created)
                continue;

            if (object.outputLine > 0)
            {
                connections[object.outputLine] = [i];
                connections[object.outputLine].push("down");
            }
            object.outputLine = null;       // fix after loading from file

            if (object.outputLineFalse > 0)
            {
                connections[object.outputLineFalse] = [i];
                connections[object.outputLineFalse].push("right or left");
            }
            object.outputLineFalse = null;  // fix after loading from file
        }
        console.debug("Found connections (outputs only): length=", connections.length);
        console.debug("Found connections (outputs only):", connections);

        // append inputs to outputs in connections[]
        for (i = 0; i < diagramField.children.length; ++i)
        {
            // omit object expcept diagramObject_N
            object = diagramField.children[i]

            if (object.visible === false)
                continue
            if (object.objectName.indexOf("diagramObject_") !== 0)
                continue
            if (object.created)
                continue;

            object.inputLinesArray.forEach(
                    function (element, index, array)
                    {
                        // if same output line index registered in connections[]
                        if (element in connections)
                        {
                            connections[element].push(i);
                            connections[element].push(object.pageIndex);
                        }
                    });
            object.inputLinesArray.length = 0;
        }
        console.debug("Found connections (with inputs): length=", connections.length);
        console.debug("Found connections (with inputs): ", connections);

        // Create connections
        connections.forEach(
            function (element, index, array)
            {
                console.debug(element[0], element[1], "->", element[2], "(", element[3], ")");

                // if input line index registered for output line index
                if (element[2] !== undefined)
                {
                    loadAndCreateItem("DiagramLine.qml", Qt.point(0, 0));
                    var line = diagramField.dynamicItem;

                    if (element[1] === "down")
                    {
                        var sender = diagramField.children[element[0]];
                        var reciever = diagramField.children[element[2]];

                        sender.outputLine = line;
                        if (reciever.filter_type === "filter")
                        {
                            line.is_filter = true;
                            // Порядок функций обязательно такой.
                            // Сначала восстанавливаем связь (она выставит к какому входу подключаемся)
                            reciever.restoreInputConnection(line, sender);
                            // Затем внутри setNext будет вызван setNewParameter для reciever.
                            // Он испоьзует номер связи которую выставили в пердыдущей операции.
                            sender.setNext(reciever);
                        } else
                        {
                            reciever.inputLinesArray.push(line);
                        }

                    }
                    else if (element[1] === "right or left")
                    {
                        diagramField.children[element[0]].outputLineFalse = line;
                        diagramField.children[element[2]].inputLinesArray.push(line);

                        if (diagramField.children[element[0]].x < diagramField.children[element[2]].x)
                            element[1] = "right";
                        else
                            element[1] = "left";

                        diagramField.children[element[0]].outputLineFalsePosition = element[1];
                    }
                    else
                    {
                        diagramField.dynamicItem.destroy();
                        diagramField.dynamicItem = null;
                    }
                }

            if (diagramField.dynamicItem)
            {
                diagramField.dynamicItem.setEndpoint1(element[1], diagramField.children[element[0]]);
                diagramField.dynamicItem.setEndpoint2(diagramField.children[element[2]]);
                diagramField.draggedItemCount += 1;
                diagramField.dynamicItem.created = true;
                diagramField.dynamicItem.pageIndex = element[3];
                diagramField.dynamicItem = null;
                recalculateLine(line);
            }
        });

        // Mark _new_ objects (without restored connections) _created_
        for (i = 0; i < diagramField.children.length; ++i)
        {
            // omit object expcept diagramObject_N
            object = diagramField.children[i]
            if (object.visible === false)
                continue
            if (object.objectName.indexOf("diagramObject_") !== 0)
                continue
            if (object.created)
                continue

            object.created = true
        }

        dumpChildren();
        console.debug("---- restoreConnections: finish");
    }

    function recalculateLine(line)
    {
        console.debug("recalculateLine:", line)
        if (line === null)
            return;

        console.debug("***** point1Type:", line.point1Type)
        line.dump()

        var xStart;     // start point of the line
        var yStart;
        var xFinish;    // end point of the line
        var yFinish;
        var xMiddle;    // absciss of the middle vertical line

        if (line.point1Type === "down")
        {
            xStart = line.endpoint1.x;
            yStart = line.endpoint1.y + linesGridSize * 2;
        }
        else if (line.point1Type === "right")
        {
            xStart = line.endpoint1.x + linesGridSize * 2;
            yStart = line.endpoint1.y;
        }
        else if (line.point1Type === "left")
        {
            xStart = line.endpoint1.x - linesGridSize * 2;
            yStart = line.endpoint1.y;
        }
        else
        {
            xStart = line.endpoint1.x;
            yStart = line.endpoint1.y;
        }

        xFinish = line.endpoint2.x;
        yFinish = line.endpoint2.y - linesGridSize*2;
        if (line.is_filter)
            yFinish = line.endpoint2.y;

        if (line.point1Type === "down")
        {
            console.debug("***** draw down line start")
            if (line.is_filter === true)
            {
                console.debug("***** it is filter");
                xStart = line.endpoint1.x + Math.abs(line.endpoint1.x - line.endpoint2.x) / 2
                line.middlePoints.length = 3;
                line.middlePoints[0] = Qt.point(xStart, line.endpoint1.y);
                line.middlePoints[1] = Qt.point(xStart, yFinish);
                line.middlePoints[2] = Qt.point(xFinish, yFinish);
            } else
            {
                if (yStart <= yFinish)
                {
                    console.debug("***** yStart <= yFinish")
                    line.middlePoints.length = 2;
                    line.middlePoints[0] = Qt.point(xStart, yFinish);
                    line.middlePoints[1] = Qt.point(xFinish, yFinish);
                }
                else
                {
                    if (xStart < xFinish)
                    {
                        if ((xFinish - xStart) < (itemWidth + linesGridSize * 4))
                            xMiddle = xFinish + itemWidth/2 + linesGridSize * 2;
                        else
                            xMiddle = xStart + itemWidth/2 + linesGridSize * 2;
                    }
                    else
                    {
                        if ((xStart - xFinish) < (itemWidth + linesGridSize * 4))
                            xMiddle = xFinish - itemWidth/2 - linesGridSize * 2;
                        else
                            xMiddle = xStart - itemWidth/2 - linesGridSize * 2;
                    }

                    line.middlePoints.length = 4;
                    line.middlePoints[0] = Qt.point(xStart, yStart);
                    line.middlePoints[1] = Qt.point(xMiddle, yStart);
                    line.middlePoints[2] = Qt.point(xMiddle, yFinish);
                    line.middlePoints[3] = Qt.point(xFinish, yFinish);
                }
            }
        }
        else if (line.point1Type === "right")
        {
            if (yStart <= yFinish)
            {
                if (xStart < xFinish)
                {
                    line.middlePoints.length = 2;
                    line.middlePoints[0] = Qt.point(xFinish, yStart);
                    line.middlePoints[1] = Qt.point(xFinish, yFinish);
                }
                else {
                    line.middlePoints.length = 3;
                    line.middlePoints[0] = Qt.point(xStart, yStart);
                    line.middlePoints[1] = Qt.point(xStart, yFinish);
                    line.middlePoints[2] = Qt.point(xFinish, yFinish);
                }
            }
            else
            {
                if (xStart < (xFinish + itemWidth / 2))
                {
                    if (((xFinish + itemWidth / 2) - xStart) < (itemWidth + linesGridSize * 2))
                        xMiddle = xFinish + itemWidth/2 + linesGridSize * 2;
                    else
                        xMiddle = xStart;

                    line.middlePoints.length = 4;
                    line.middlePoints[0] = Qt.point(xStart, yStart);
                    line.middlePoints[1] = Qt.point(xMiddle, yStart);
                    line.middlePoints[2] = Qt.point(xMiddle, yFinish);
                    line.middlePoints[3] = Qt.point(xFinish, yFinish);
                }
                else
                {
                    line.middlePoints.length = 3;
                    line.middlePoints[0] = Qt.point(xStart, yStart);
                    line.middlePoints[1] = Qt.point(xStart, yFinish);
                    line.middlePoints[2] = Qt.point(xFinish, yFinish);
                }
            }
        }
        else if (line.point1Type === "left")
        {
            if (yStart <= yFinish)
            {
                if (xStart < xFinish)
                {
                    line.middlePoints.length = 3;
                    line.middlePoints[0] = Qt.point(xStart, yStart);
                    line.middlePoints[1] = Qt.point(xStart, yFinish);
                    line.middlePoints[2] = Qt.point(xFinish, yFinish);
                }
                else
                {
                    line.middlePoints.length = 2;
                    line.middlePoints[0] = Qt.point(xFinish, yStart);
                    line.middlePoints[1] = Qt.point(xFinish, yFinish);
                }
            }
            else {
                if (xStart < (xFinish - itemWidth / 2))
                {
                    line.middlePoints.length = 3;
                    line.middlePoints[0] = Qt.point(xStart, yStart);
                    line.middlePoints[1] = Qt.point(xStart, yFinish);
                    line.middlePoints[2] = Qt.point(xFinish, yFinish);
                }
                else
                {
                    if ((xStart - (xFinish - itemWidth / 2)) < (itemWidth + linesGridSize * 2))
                        xMiddle = xFinish - itemWidth / 2 - linesGridSize * 2;
                    else
                        xMiddle = xStart;

                    line.middlePoints.length = 4;
                    line.middlePoints[0] = Qt.point(xStart, yStart);
                    line.middlePoints[1] = Qt.point(xMiddle, yStart);
                    line.middlePoints[2] = Qt.point(xMiddle, yFinish);
                    line.middlePoints[3] = Qt.point(xFinish, yFinish);
                }
            }
        }
        else
        {
            line.middlePoints.length = 2;
            line.middlePoints[0] = Qt.point(xStart, yStart);
            line.middlePoints[1] = Qt.point(xFinish, yFinish);
        }
        line.updatePrimitives();

        scriptGenerator.scriptModifiedFromQML();
    }

    function alignItem(item) {
        var x = Math.round(item.x / objectsGridSize2x) * objectsGridSize2x;
        var y = Math.round(item.y / objectsGridSize2x) * objectsGridSize2x;
        item.x = x;
        item.y = y;
    }

    function getAlignedPoint(point) {
        var x = Math.round(point.x / objectsGridSize2x) * objectsGridSize2x;
        var y = Math.round(point.y / objectsGridSize2x) * objectsGridSize2x;
        return Qt.point(x, y)
    }

    // process mouse buttons press with modificators
    function itemSelected(item, mouseEvent) {
        console.debug(arguments.callee.name, item, mouseEvent)
        selectionChanged(item, mouseEvent, item.selected)
    }

    // process item dragging with pressed mouse button
    function itemMoved(item, point) {
        console.debug("itemMoved:", item, point);
        var deltaX = item.x - item.startPoint.x;
        var deltaY = item.y - item.startPoint.y;
        selectionMoved(item, deltaX, deltaY);
        scriptGenerator.scriptModifiedFromQML();
    }

    // process mouse buttons release on item
    function itemReleased(item)
    {
        alignItem(item);
        console.debug("itemReleased:", item, x, y);
        var deltaX = item.x - item.startPoint.x
        var deltaY = item.y - item.startPoint.y

        selectionReleased(item);

        if (item.filter_type === "filter")
        {
            item.filterGenerateNeewResult();
        }

        // recalculate lines for every selected item
        var linesToRecalculate = []
        for (var i = 0 ; i < diagramField.children.length; ++i)
        {
            // process only diagramObject_N
            var object = diagramField.children[i]
            if (object.visible === false)
                continue
            if (object.objectName.indexOf("diagramObject_") === -1)
                continue
            if (object.selected === false)
                continue
            if (object.created === false)
                continue

            if (object.outputLine !== null)
            {
                object.outputLine.setEndpoint1("down", object);
                linesToRecalculate.push(object.outputLine);
            }

            if (object.outputLineFalse !== null)
            {
                // automatic correct outputLineFalsePosition for Condition outputs
                if (2 <= object.actionOutputCount)
                {
                    var outConnectedObject = findItemByInputLine(object.outputLineFalse);
                    if (object.x <= outConnectedObject.x)
                        object.outputLineFalsePosition = "right";
                    else
                        object.outputLineFalsePosition = "left";
                }

                object.outputLineFalse.setEndpoint1(object.outputLineFalsePosition, object);

                linesToRecalculate.push(object.outputLineFalse);
            }
            // recalculate input lines
            console.log("input array size: ", object.inputLinesArray.length);
            object.inputLinesArray.forEach(function (line, index, array)
                {
                    // automatic correct outputLineFalsePosition for Condition outputs
                    var inConnectedObject = findItemByOutputLine(line);
                    if ((inConnectedObject !== undefined) &&
                        (2 <= inConnectedObject.actionOutputCount) &&
                        (inConnectedObject.outputLineFalse === line))
                    {
                        if (inConnectedObject.x <= object.x)
                            inConnectedObject.outputLineFalsePosition = "right";
                        else
                            inConnectedObject.outputLineFalsePosition = "left";
                        line.setEndpoint1(
                                    inConnectedObject.outputLineFalsePosition,
                                    inConnectedObject);
                    }

                    line.setEndpoint2(object);
                    linesToRecalculate.push(line);
            });
        }

        linesToRecalculate = linesToRecalculate.filter(unique);
        console.debug("linesToRecalculate:", linesToRecalculate);
        linesToRecalculate.forEach(function (line, index, array) {
            recalculateLine(line);
        });
        if (deltaX !== 0 || deltaY !== 0) {
            scriptGenerator.pushState(diagramContainer)
        }
    }

    function dragPosChanged(item, dragPoint) {
        if (dragTarget === null)
        {
            dragTarget = item.dragItem
            dragTarget.hasError = false

            if (item.objectName.indexOf("dragPointFalse") === 0)
                dragTargetType = item.pointPosition;
            else if (item.objectName.indexOf("dragPointTrue") === 0)
                dragTargetType = "down"
            else
                dragTargetType = ""
        }

        console.debug('dragPosChanged: dragPoint: ', dragPoint)
        var pointInDiagram = mapFromItem(item, dragPoint.x, dragPoint.y)
        console.debug('dragPosChanged: pointInDiagram:', pointInDiagram.x, pointInDiagram.y)
        dropPosChanged(dragTarget, Qt.point(pointInDiagram.x, pointInDiagram.y))

        dropTarget = null
        for (var i = 0 ; i < diagramField.children.length; ++i) {
            // process only diagramObject_N
            var object = diagramField.children[i]
            if (object.visible === false)
                continue;
            if (object.objectName.indexOf("diagramObject_") === -1)
                continue;
            if (object.pageIndex !== pageIndex)
                continue;
            if (object.created === false)
                continue;

            // find object "diagramItemRect"
            var objectItemRect;
            for (var j = 0; j < object.children.length; ++j)
                if (object.children[j].objectName === "diagramItemRect") {
                    objectItemRect = object.children[j];
                    break;
                }

            var pointInObjectItemRect =
                    mapToItem(objectItemRect, pointInDiagram.x, pointInDiagram.y)

            // check if we're hovering the objectItemRect
            var isObjectItemRectHovered = objectItemRect.contains(
                        Qt.point(pointInObjectItemRect.x, pointInObjectItemRect.y))

            // view_type = 1 - Соответсвтует GetParameter, коннектиться к которому мы запрещаем
            // Для скриптов это будет undefined значение.
            if (isObjectItemRectHovered && (Vtypes.view_types.get_param !== object.view_type))
            {
                last_mapped_pos_x = pointInObjectItemRect.x;
                last_mapped_pos_y = pointInObjectItemRect.y;
                dropTarget = object
            }

            console.debug('dragPosChanged: pointInObjectItemRect:',
                        i, object.objectName,
                        pointInObjectItemRect.x, pointInObjectItemRect.y,
                        isObjectItemRectHovered)
            scriptGenerator.scriptModifiedFromQML();
        }
    }

    // Функция вызывается когда закончили перетаскивать соединительную стрелку.
    // В этом случае уже установлен/либо остался null dropTarget.
    // Таким образом у нас здесь уже есть как раз информация о соединяемых объектах.
    // Для них нужно установить связь.
    function dragFinished(mouse) {
        dropFinished()

        console.debug("dragFinished:", dragTarget, dropTarget, mouse)

        var line = diagramField.dynamicItem;

        if ((dropTarget === null) || (dropTarget == dragTarget) || (dropTarget.actionType !== "action"))
        {
            line.visible = false;
            line.destroy()
            diagramField.draggedItemCount -= 1;
        }
        else
        {
            dropTarget.hasError = false

            var lineToDestroy = null;
            if ((dragTargetType === "right") || (dragTargetType === "left"))
            {
                line.setEndpoint1(dragTargetType, dragTarget)
                if (dragTarget.outputLineFalse !== null)
                {
                    lineToDestroy = dragTarget.outputLineFalse;
                    console.debug("Destroy previous false-connection. Schedule destruction of the line:",
                                lineToDestroy);
                }
                dragTarget.outputLineFalse = line
                dragTarget.outputLineFalsePosition = dragTargetType;
            }
            else if (dragTargetType === "down")
            {
                line.setEndpoint1("down", dragTarget)
                if (dragTarget.outputLine !== null)
                {
                    lineToDestroy = dragTarget.outputLineFalse;
                    console.debug("Destroy previous true-connection. Schedule destruction of the line:",
                                lineToDestroy);
                }
                dragTarget.outputLine = line
            }

            // Тут важен порядок. Перед тем как установить endpoint необходимо выставить connection Object
            if (dropTarget.filter_type === "filter")
            {
                line.is_filter = true;
                dropTarget.setConnectionObject(dragTarget, last_mapped_pos_x, last_mapped_pos_y);
            }

            // После этого подготовить line
            line.setEndpoint2(dropTarget);

            // И вставить ее в массив входных линий.
            if (dropTarget.filter_type === "filter") {
                dropTarget.addInputLine(line);
            } else {
                dropTarget.inputLinesArray.push(line);
            }

            if (lineToDestroy !== null) {
                console.debug("Destroy previous connection. Line:",
                            lineToDestroy);
                lineIsAboutToDestroy(lineToDestroy);
                lineToDestroy.destroy();
            }

            recalculateLine(line);
            selectionChanged(line, mouse, false)

            dropTarget.selected = true;
            itemReleased(dropTarget);
        }

        // Установим связь, если работаем с фильтром.
        if ((dropTarget !== null ) && ("filter" === dropTarget.filter_type))
            dragTarget.setNext(dropTarget);

        dragTarget = null
        dropTarget = null

        recolorChains();
        scriptGenerator.pushState(diagramContainer)
        dumpChildren();
    }

    function lineIsAboutToDestroy(line) {
        console.debug("lineIsAboutToDestroy", line, line.objectName)

        var i,j
        for (i = 0; i < diagramField.children.length; ++i) {
            var object = diagramField.children[i]

            if (object.visible === false)
                continue;
            if (object.objectName.indexOf("diagramObject_") !== 0)
                continue;
            if (object.created === false)
                continue;

            if (object.outputLine === line) {
                object.outputLine = null
                console.debug(object.objectName, "outputLineFalse cut off")
            }

            if (object.outputLineFalse === line) {
                object.outputLineFalse = null
                console.debug(object.objectName, "outputLineFalse cut off")
            }

            var lineIndexToRemove = -1;
            for (j=0; j<object.inputLinesArray.length; ++j) {
                var inputLine = object.inputLinesArray[j];
                if (inputLine === line) {
                    lineIndexToRemove = j
                    console.debug(object.objectName, j, "is about to cut off")
                    break;
                }
            }
            if (lineIndexToRemove !== -1) {
                object.inputLinesArray.splice(lineIndexToRemove, 1)
                console.debug(object.objectName, "inputLinesArray cut off:",
                              lineIndexToRemove)
                console.debug(object.objectName,
                              "inputLinesArray:", object.inputLinesArray)
            }
        }

        line.visible = false
    }

    function itemIsAboutToDestroy(item) {
        console.debug("itemIsAboutToDestroy", item);
        item.dump();

        var i;
        // destroy parametersDialog
        if (item === parametersDialog) {
            parametersDialog = null;
        }
        // destroy ActionButtons
        else if (item.objectName.indexOf("buttonItem") === 0)
        {
            ;
        }
        // destroy connected lines
        else
        {
            var linesToDestroy = []
            linesToDestroy = linesToDestroy.concat(
                        item.inputLinesArray,
                        item.outputLine,
                        item.outputLineFalse);

            console.debug(linesToDestroy);
            linesToDestroy.forEach(function (line, index, array) {
                if (line !== null) {
                    console.debug("Destroying line:", line);
                    line.destroyItem();
                }
            });
        }

        console.debug("itemIsAboutToDestroy finish");
    }

    //! For each line look through every diagramObject_ for:
    //! 1) null object connection lines that is equal to destroyed one
    //! 3) destroy line
    function destroyLines(linesToDestroy) {
        console.debug("destroyLines():", linesToDestroy);
        var i, line, object;

        // 1) null...
        linesToDestroy.forEach(function (line, index, array) {
            line.visible = false;
            for (i = 0; i < diagramField.children.length; ++i) {
                object = diagramField.children[i];
                if (object.objectName.indexOf("diagramObject_") === -1)
                    continue;

                if (object.outputLine === line)
                    object.outputLine = null;

                if (object.outputLineFalse === line)
                    object.outputLineFalse = null;

                object.inputLinesArray.forEach(function (inputLine, index, array) {
                    if (inputLine === line)
                        array[index] = null;
                });
                // remove null indexes from object.inputLinesArray
                object.inputLinesArray =
                        object.inputLinesArray.filter(function (element) {
                            return element !== null;
                        });
            };
        });

        // 2) destroy...
        linesToDestroy.forEach(function (line, index, array) {
            line.destroy();
        });
    }

    //! Destroy diagramObgects and diagramLines of \a items array
    function destroyItems(items) {
        console.debug("destroyItems()", items);
        if (items.length === 0)
            return;

        // 1) collect lines of every item to destroy...
        var linesToDestroy = [];
        items.forEach(function (item, index, array) {
            // item is object. get item lines
            if (item.objectName.indexOf("diagramObject_") === 0)
            {
                linesToDestroy = linesToDestroy.concat(
                            item.inputLinesArray,
                            item.outputLine,
                            item.outputLineFalse);
            }
            // item is line. get it
            else
            {
                linesToDestroy.push(item);
            }
        });
        // find not null and unique
        linesToDestroy =
                linesToDestroy.filter(function (line, index, self) {
                    return (line !== null)
                            && (self.indexOf(line) === index);
                });
        destroyLines(linesToDestroy);

        items.forEach(function (item, index, array) {
            item.destroy();
        });
        scriptGenerator.pushState(diagramContainer)
        console.debug("destroyItems() finish");
        dumpChildren();
    }

    function removePage(removeIndex) {
        // process and objects and lines
        for (var i = 0; i < diagramField.children.length; ++i) {
            var object = diagramField.children[i];
            if (object.visible === false)
                continue;
            if (object.objectName.indexOf("diagram") !== 0)
                continue;
            if (object.created === false)
                continue;

            if (object.pageIndex === removeIndex)
                object.visible = false;
            else if (removeIndex < object.pageIndex)
                --object.pageIndex;
        }
        scriptGenerator.scriptModifiedFromQML();
    }

    function movePage(fromIndex, toIndex) {
        // swap page indexes of objects and lines
        for (var i = 0; i < diagramField.children.length; ++i) {
            var object = diagramField.children[i];
            if (object.visible === false)
                continue;
            if (object.created === false)
                continue;
            if ((object.objectName.indexOf("diagramObject_") !== 0)
                && (object.objectName.indexOf("diagramLine_") !== 0))
            {
                continue;
            }

            if (object.pageIndex === fromIndex)
                object.pageIndex = toIndex;
            else if (object.pageIndex === toIndex)
                object.pageIndex = fromIndex;
        }
        scriptGenerator.scriptModifiedFromQML();
    }

    //! obtain all selected objects and lines of current page
    function getSelectedArray() {
        var selectedArray = [];

        for (var i = 0 ; i < diagramField.children.length; ++i) {
            var object = diagramField.children[i];
            if (object.visible === false)
                continue;
            if (object.pageIndex !== pageIndex)
                continue;
            if (object.selected === false)
                continue;
            if (object.created === false)
                continue;

            // search for objects
            if ((object.objectName.indexOf("diagramObject_") === 0)
                    && (object.selected === true))
            {
                selectedArray.push(object);
            }
            // search for lines
            if ((object.objectName.indexOf("diagramLine_") === 0)
                    && (object.state === "selected"))
            {
                selectedArray.push(object);
            }
        }
        console.debug("getSelectedArray()", pageIndex, selectedArray);
        return selectedArray;
    }

    //! repaint all chains with its color
    function recolorChains() {
        var objectsArray = [];
        for (var i = 0; i < diagramField.children.length; ++i) {
            var object = diagramField.children[i];
            if (object.visible === false)
                continue;
            if (object.objectName.indexOf("diagramObject_") !== 0)
                continue;
            if (object.actionType.indexOf("Event") === -1)
                continue;
            if (object.created === false)
                continue;

            objectsArray.push(object);
        }

        objectsArray.forEach(function (element, index, array) {
            console.debug(arguments.callee.name,
                          element.actionName, element.chainColor);
            setChainColor(element, element.chainColor);
        });
    }

    //! set chainColor for every object which is connected to \a item
    function setChainColor(item, color) {
        var objectsArray = [];
        objectsArray.push(item);
        var selectedArray = [];
        var object, connectedObject;
        while (objectsArray.length) {
            console.debug(objectsArray)

            object = objectsArray.pop();
            if (object === undefined)
                continue;

            object.chainColor = color;
            console.debug(arguments.callee.name, object.actionName, object.chainColor);

            // find next objects connected below
            if (object.outputLine !== null) {
                connectedObject = findItemByInputLine(object.outputLine);
                if (selectedArray.indexOf(connectedObject) === -1) {
                    selectedArray.push(connectedObject);
                    objectsArray.push(connectedObject);
                }
            }

            if (object.outputLineFalse !== null) {
                connectedObject = findItemByInputLine(object.outputLineFalse);
                if (selectedArray.indexOf(connectedObject) === -1) {
                    selectedArray.push(connectedObject);
                    objectsArray.push(connectedObject);
                }
            }

        }
        scriptGenerator.scriptModifiedFromQML();
    }

    //! Hide all diagramObject_ and diagramLine_
    function hideAllFieldObjects() {
        console.debug("---- hideAllFieldObjects()");
        for (var i = 0; i < diagramField.children.length; ++i) {
            var object = diagramField.children[i];
            if (object.objectName.indexOf("diagramObject_") === 0)
                object.visible = false;
            if (object.objectName.indexOf("diagramLine_") === 0)
                object.visible = false;
        }
        dumpChildren();
    }

    //! Manual destroying all of the hidden diagram objects.
    function destroyHiddenFieldObjects() {
        console.debug("---- destroyHiddenFieldObjects()");
        for (var i = 0; i < diagramField.children.length; ++i) {
            var object = diagramField.children[i];
            if ((object.objectName.indexOf("diagramObject_") !== 0)
                && (object.objectName.indexOf("diagramLine_") !== 0))
            {
                continue;
            }

            if (object.visible === false) {
                object.destroy();
                continue;
            }
        }
        dumpChildren();
    }

    //! Go through all objects and delete all variables aren't used in diagram
    function removeUnusedVariables()
    {
        var usedVarNames = [];
        // process every diagramObject_N
        for (var i = 0; i < diagramField.children.length; ++i)
        {
            var object = diagramField.children[i];

            if (object.visible === false)
                continue
            if (object.objectName.indexOf("diagramObject_") !== 0)
                continue
            if (object.created === false)
                continue;

            // process every parameter
            var count = object.parametersModel.count
            for (var k = 0; k < count; ++k)
                console.log(object.parametersModel.get(k).varName, object.parametersModel.get(k).varCategory);

            for (var j = 0; j < count; ++j)
            {
                var param = object.parametersModel.get(j);

                if (param.varName === "")       // skip empty
                    continue
                if (param.paramType === categoryList) // skip param lists
                    continue

                // collect every new varName
                var name = param.varName;
                if (usedVarNames.indexOf(name) === -1)
                    usedVarNames.push(name);
            }
        }
        console.debug("removeUnusedVariables(): usedVar", usedVarNames);
        scriptGenerator.removeUnusedVariables(usedVarNames);
    }

    function clearState() {
        selectionChanged(diagramField, undefined, false);
        hideAllFieldObjects()
        destroyHiddenFieldObjects()
    }

    //! Is category (c) contained in string of categories (c1|c2|c3...)?
    function isMatched(p, params) {
        var splitParams = params.split("|");
        for (var i = 0; i < splitParams.length; ++i)
        {
            if(p === splitParams[i])
                return true;
        }
        return false;
    }

    // get id from proxy varsProxyModel
    function getVarIdFromProxyModel(proxyModel, parameter)
    {
        var k;
        for (k = 0; k < proxyModel.count; ++k)
        {
            if (proxyModel.get(k, "varName") === parameter.varName &&
                isMatched(proxyModel.get(k, "type"), parameter.paramType) &&
                proxyModel.get(k, "category") === parameter.varBaseCategory)
                return k;

        }
        return -1;
    }

    //! Check diagram for correctness and
    //! prepare parameters for Script Generator
    function checkDiagram()
    {
        console.debug("checkDiagram: start");
        var i, j, k, il;
        var chainCount = 0;

        removeUnusedVariables();

        var objects=[];

        for (i = 0; i < diagramField.children.length; ++i)
        {
            var object = diagramField.children[i];
            if (object.visible === false)
                continue
            if (object.objectName.indexOf("diagramObject_") !== 0)
                continue
            if (object.created === false)
                continue;
            objects.push(object);
        }

        // Инициализируем нужные переменные
        for (i = 0; i < objects.length; ++i)
        {
            var object = objects[i];
            if (object.actionName === "DeltaFilter")
                object.generatePreviousValue();
        };

        // process every diagramObject_N
        for (i = 0; i < objects.length; ++i)
        {
            var object = objects[i];

            if ((object.actionType.indexOf("systemEvent") !== -1) || (object.actionType.indexOf("userEvent") !== -1))
                ++chainCount;

            // Проверка связей, что линии у нас попадают куда надо
            if (object.outputLine !== null && !checkLine(object.outputLine))
            {
                showError(object,
                           qsTr("The output of '%1' has to be assinged").
                           arg(qsTranslate(object.actionName, object.actionName)));
                return false;
            }

            for (il = 0; il < object.inputLinesArray.length ; ++il)
                if ( !checkLine(object.inputLinesArray[il]))
                {
                    showError(object,
                               qsTr("The input of '%1' has to be assinged").
                               arg(qsTranslate(object.actionName, object.actionName)));
                    return false;
                }

            if (object.filter_type !== "filter")
            {
                if (((object.actionOutputCount === 1) && (object.outputLine === null)) ||
                        ((object.actionOutputCount === 2) && ((object.outputLine === null) || (object.outputLineFalse === null))))
                {
                    showError(object,
                              qsTr("The output of '%1' has to be assinged").
                              arg(qsTranslate(object.actionName, object.actionName)));
                    return false;
                }
            } else
            {
                // Для фильтров будет отдельная проверка
                // 1. Сначала проеряем имя объекта. Определим его тип.
                // 2. Проведемм разный разбор.
                // 2.1. Если у нас Параметр. То у него должна быть 1 связь. Либо input либо output.
                // 2.2. Если у нас Фильтр, то у него должны быть обязатльно 2 связи и Input и Output.
                // 2.3. Если у нас Фильтр с 2-мя входами, то у него должны быть обязатльно 3 связи и 2 Input и Output.

                var blocks_types = ["parameter", "filter", "parameter", "filter_2_params"]; // getparam, filter, saveparam, filter_two_out
                var block_type = blocks_types[object.view_type - 1];

                if (block_type === "parameter")
                {
                    if ((object.outputLine === null) && (object.inputLinesArray.length === 0))
                    {
                        showError(object,
                                  qsTr("The output or input of '%1' has to be assinged").
                                  arg(qsTranslate(object.actionName, object.actionName)));
                        return false;
                    }
                } else if (block_type === "filter")
                {
                    if ((object.outputLine === null) || (object.inputLinesArray.length === 0))
                    {
                        showError(object,
                                  qsTr("The output and input of '%1' has to be assinged").
                                  arg(qsTranslate(object.actionName, object.actionName)));
                        return false;
                    }
                } else if (block_type === "filter_2_params")
                {
                    if ((object.outputLine === null) || (object.inputLinesArray.length !== 2))
                    {
                        showError(object,
                                  qsTr("The output and all input of '%1' has to be assinged").
                                  arg(qsTranslate(object.actionName, object.actionName)));
                        return false;
                    }
                }
            }

            if ((object.actionType === "action") && (object.inputLinesArray.length === 0))
            {
                showError(object,
                          qsTr("The input of '%1' has to be assinged").
                          arg(qsTranslate(object.actionName, object.actionName)));
                return false;
            }

            //Проверим фильтры тарировочных таблиц. Есть ли хотябы 1 элемент в таблице тарировки
            if (object.need_check_tar_table)
            {
                switch(object.view_type)
                {
                    case Vtypes.view_types.one_input_filter:
                    case Vtypes.view_types.two_inputs_filter:
                        if (object.origin_param_count >= object.parametersModel.count)
                        {
                            showError(object,
                                      qsTr("No points in calibration table. Please define at least one point"));
                            return false;
                        }
                        break;
                    default: break;
                }
            }

            object.parametersNames.length = 0;
            object.parametersVarIds.length = 0;
            object.parametersVarCategories.length = 0;

            // process every parameter
            console.debug("process every parameter");
            for (j = 0; j < object.parametersModel.count; ++j)
            {
                var param = object.parametersModel.get(j);
                console.debug("current: ", j, object.actionName, param.paramName, param.varName, param.paramType)

                object.parametersNames[j] = param.paramName;
                if (param.varName === "")
                {
                    showError(object,
                              qsTr("Parameter '%1' in action '%2' is not defined ")
                              .arg(qsTranslate(object.actionName, param.paramName))
                              .arg(qsTranslate(object.actionName, object.actionName)));
                    return false;
                }

                if (param.paramType === categoryList)
                {
                    // search index of list param
                    var _name = param.varName;
                    for (k = 0; k < param.paramList.count; ++k)
                    {
                        if (param.paramList.get(k).varName === _name)
                        {
                            object.parametersVarIds[j] = param.paramList.get(k).value;
                            object.parametersVarCategories[j] = categoryList;
                            break;
                        }
                    }

                    if (k !== param.paramList.count) // varName was found
                        continue;

                    // Фильтр с 2-мя входами Для топлива особенный. Для него разрешаем задавать значения
                    if (object.view_type === Vtypes.view_types.two_inputs_filter)
                        continue;

                    showError(object,
                              qsTr("Variable '%1' not found in paramList of parameter '%2'")
                              .arg(qsTranslate(object.actionName, param.varName))
                              .arg(qsTranslate(object.actionName, param.paramName)));
                    return false;
                }

                // Рассмотрим категорию canIndex
                if (param.paramType === categoryCanIndex)
                {
                    object.parametersVarIds[j] = param.varName;
                    object.parametersVarCategories[j] = categoryCanIndex;
                    continue;
                }

                // Try to compile script
                if (param.paramType === typeScript)
                {
                    var res = scriptGenerator.compileScript(param.varName)
                    if (res !== 0)
                    {
                        showError(object,
                                  qsTr("Script '%1' compilation error = %2")
                                  .arg(qsTranslate(object.actionName, param.varName))
                                  .arg(res));
                        return false;
                    }
                }

                // search in const vars
                var id = getVarIdFromProxyModel(constVarsProxyModel, param)
                if (id !== -1)
                {
                    object.parametersVarIds[j] = id;
                    object.parametersVarCategories[j] = categoryConst;
                    continue;
                }

                // search in global vars
                id = getVarIdFromProxyModel(globalVarsProxyModel, param)
                if (id !== -1)
                {
                    object.parametersVarIds[j] = id;
                    object.parametersVarCategories[j] = categoryGlobal;
                    continue;
                }

                // search in localConst vars
                id = getVarIdFromProxyModel(localConstVarsProxyModel, param)
                if (id !== -1)
                {
                    object.parametersVarIds[j] = id;
                    object.parametersVarCategories[j] = categoryLocalConst;
                    continue;
                }

                // search in local vars
                id = getVarIdFromProxyModel(localVarsProxyModel, param)
                if (id !== -1)
                {
                    object.parametersVarIds[j] = id;
                    object.parametersVarCategories[j] = categoryLocal;
                    continue;
                }

                showError(object,
                          qsTr("Variable '%1' not found")
                          .arg(qsTranslate(object.actionName, param.varName)));
                return false;
            }
            console.debug(object.actionName, object.parametersNames, object.parametersVarIds, object.parametersVarCategories)
        }

        console.debug("checkDiagram: Done. OK");
        return true;
    }

    function prepareForSaving()
    {
        console.debug(arguments.callee.name, "Start");
        // process every diagramObject_N
        var i
        for (i = 0; i < diagramField.children.length; ++i)
        {
            var object = diagramField.children[i]
            if (object.visible === false)
                continue
            if (object.objectName.indexOf("diagramObject_") !== 0)
                continue
            if (object.created === false)
                continue;

            object.prepareForSaving()
        }
        console.debug(arguments.callee.name, "Done");
    }

    function varNameChanged(oldName, newName) {
        console.debug("varNameChanged()", oldName, "->", newName);
        // process every diagramObject_N
        var i, j;
        for (i = 0; i < diagramField.children.length; ++i) {
            var object = diagramField.children[i];
            if (object.visible === false)
                continue;
            if (object.objectName.indexOf("diagramObject_") !== 0)
                continue;
            if (object.created === false)
                continue;

            // process every parameter
            for (j = 0; j < object.parametersModel.count; ++j) {
                var param = object.parametersModel.get(j);
                if (param.varName === oldName) {
                    param.varName = newName;
                    console.debug(object.objectName, param.paramName, ":",
                                  oldName, "->", newName);
                }
            }
        }
        scriptGenerator.pushState(diagramContainer)
    }

    function showError(object, errorString) {
        console.error(errorString);
        object.hasError = true;
        messageDialog.text = errorString;
        messageDialog.open();
    }

    width: diagramFieldWidth
    height: diagramFieldHeight

    transform: [
        Scale {
            origin.x: width/2;
            origin.y: height/2;
            xScale: diagramField.diagramScaleFactor
            yScale: diagramField.diagramScaleFactor
        },
        Translate {
            x: diagramCenterPointX - width/2
            y: diagramCenterPointY - height/2
        }
    ]

    Component.onCompleted: {
        diagramContainer.centerDiagramField()
    }

    Rectangle {
        id: selectRect
        objectName: "selectRect"

        property point startPoint
        property point mousePoint

        border {width: 2; color: colorSelected }
        color: Qt.lighter(colorSelected, 1.6)
        z: zDiagramSelectRect
        opacity: 0.5
        visible: (startPoint !== mousePoint)

        function recalc() {
            x = Math.min(startPoint.x, mousePoint.x);
            y = Math.min(startPoint.y, mousePoint.y);
            width  = Math.abs(mousePoint.x - startPoint.x);
            height = Math.abs(mousePoint.y - startPoint.y);

            console.debug(startPoint, mousePoint,
                          selectRect.x, selectRect.y,
                          selectRect.width, selectRect.height)
        }

        onStartPointChanged: { mousePoint = startPoint; }
        onMousePointChanged: { recalc(); }

        Component.onCompleted: { startPoint = mousePoint = Qt.point(0, 0); }
    }

    MouseArea {
        id: mouseArea
        objectName: "mouseArea"

        anchors.fill: parent
        acceptedButtons: Qt.LeftButton

        onPressed: {
            diagramContainer.forceActiveFocus();
            if (mouse.button === Qt.LeftButton) {
                console.debug(parent.objectName, "onPressed:");
                selectRect.startPoint = Qt.point(mouseX, mouseY);
                selectionChanged(diagramField, mouse, false);
            }
        }
        onPositionChanged: {
            if (mouse.buttons & Qt.LeftButton) {
                if (mouseArea.pressed) {
                    console.debug(parent.objectName, "onPositionChanged:");
                    selectRect.mousePoint = Qt.point(mouseX, mouseY);
                }
            }
        }
        onReleased: {
            if (mouse.button === Qt.LeftButton) {
                console.debug(parent.objectName, "onReleased:");
                selectionFinished(Qt.rect(selectRect.x,
                                          selectRect.y,
                                          selectRect.width,
                                          selectRect.height));
                selectRect.startPoint = Qt.point(mouseX, mouseY);
            }
        }
    }

    Rectangle {
        id: fieldVisualizer
        objectName: "fieldVisualizer"

        property int gridWidth: 2
        property int gridPattern: Qt.Dense5Pattern

        anchors.fill: parent
        z: zDiagramField
        color: colorDiagramField
        antialiasing: true

        Grid {
            id: verticalLines
            anchors.fill: parent
            anchors.leftMargin: objectsGridSize2x * 1.5 + 1
            columns: parent.width / (objectsGridSize2x * 2) + 1
            rows: 1
            spacing: objectsGridSize2x * 2 - fieldVisualizer.gridWidth

            Repeater {
                model: parent.columns
                Canvas {
                    width: fieldVisualizer.gridWidth
                    height: diagramField.height

                    onPaint: {
                        var ctx = getContext('2d');
                        ctx.save();
                        ctx.fillStyle = ctx.createPattern(colorDiagramGrid, fieldVisualizer.gridPattern);
                        ctx.fillRect(0, 0, width, height);
                        ctx.restore();
                    }
                }
            }
        }
        Grid {
            id: horizontalLines
            anchors.fill: parent
            anchors.topMargin: objectsGridSize2x * 1.5 + 1
            columns: 1
            rows: parent.height / (objectsGridSize2x * 2) + 1
            spacing: objectsGridSize2x * 2 - fieldVisualizer.gridWidth

            Repeater {
                model: parent.rows
                Canvas {
                    width: diagramField.width
                    height: fieldVisualizer.gridWidth

                    onPaint: {
                        var ctx = getContext('2d');
                        ctx.save();
                        ctx.fillStyle = ctx.createPattern(colorDiagramGrid, fieldVisualizer.gridPattern);
                        ctx.fillRect(0, 0, width, height);
                        ctx.restore();
                    }
                }
            }
        }
    }

    MessageDialog {
        id: messageDialog
        objectName: "messageDialog"
        title: qsTr("GalileoSkyScripter")
        text: qsTr("It's so cool that you are using Qt Quick.")
        buttons: StandardButton.Ok
        modality: Qt.ApplicationModal
        flags: dialogWindowFlags
    }
}
