import { QWebChannel, QWebChannelTransport } from 'qwebchannel';
import Logger from './logger';
import Queue from './queue';

class Api {
  /**
   * Queue instance
   */
  private _queue: Queue;

  /**
   * QWebChannel instance
   */
  private _channel?: QWebChannel = undefined;

  /**
   * The latest command execution timeout
   */
  private _timer?: number = undefined;

  /**
   * Api resolve callback
   */
  private _resolve?: (data: string) => void;

  constructor(queue: Queue) {
    this._queue = queue;

    this._init();
  }

  private _init = (): void => {
    this._createWebChannel();
  };

  private _createWebChannel = (): void => {
    const qt: QWebChannelTransport = (<any>window)['qWebChannel'];

    if (!qt) {
      Logger.error('fatal error: Qt is not available');
      return;
    }

    Logger.log(`creating web channel with ${JSON.stringify(qt)}`);

    new QWebChannel(qt.webChannelTransport, this._handleChannelReady);
  };

  private _handleChannelReady = (channel: QWebChannel): void => {
    Logger.log(`created channel ${JSON.stringify(channel)}`);

    this._channel = channel;
    (<any>window).setResponseCommand = this._handleResponse;
  };

  private _handleResponse = (
    cmd: string,
    response: string,
    responseId: string
  ): void => {
    Logger.log(
      `response id: ${responseId}, cmd: ${cmd}, response: ${response}`
    );

    const { current } = this._queue;

    if (current?.id !== responseId) {
      Logger.warn(
        `
        response has wrong id. 
        current_id: ${current?.id}, 
        response_id: ${responseId}
        `
      );
      return;
    }

    if (this._resolve) {
      this._resolve(response);
      this._resolve = undefined;
    }

    if (this._timer !== undefined) {
      clearTimeout(this._timer);
    }
  };

  public send = (
    id: string,
    request: string,
    timeout: number
  ): Promise<string> => {
    return new Promise<string>((resolve, reject) => {
      this._timer = window.setTimeout(reject, timeout);
      this._resolve = resolve;

      if (!this._channel) {
        Logger.error('cannot send command: no channel');
        return reject();
      }

      if (!this._channel?.objects.webobj.sendCommand) {
        Logger.error('channel has no sendCommand method');
        return reject();
      }

      this._channel?.objects.webobj.sendCommand(request, id);
    });
  };
}

export default Api;
