import React, { useMemo, useState } from 'react';
import { FormProvider, SubmitHandler, useForm } from 'react-hook-form';
import { IForm } from '../../types/form';
import { getDateTimeString } from '../../util';
import Commander from '../../util/commander';
import Logger from '../../util/logger';
import Button from '../Button';
import CommandLog from '../CommandLog';
import Input from '../Input';
import SubmitButton from '../SubmitButton';
import { FormStyled } from './styled';

interface IFormProps {
  commander: Commander;
}

const Form: React.FC<IFormProps> = ({ commander }) => {
  const [log, setLog] = useState<string[]>([]);

  const methods = useForm<IForm>({
    defaultValues: {
      value: '',
    },
  });

  const { handleSubmit } = methods;

  const handleResponse = (response?: string): void => {
    setLog((prevValue) => {
      return prevValue.concat(`${getDateTimeString()} << ${response}`);
    });
  };

  const handleFormSubmit: SubmitHandler<IForm> = async (data) => {
    const { value } = data;

    if (!value) {
      return;
    }

    commander
      .send({ request: value })
      .then(handleResponse)
      .catch(() => {
        Logger.error(`command '${value}' execution failed`);
      });

    setLog((prevValue) =>
      prevValue.concat(`${getDateTimeString()} >> ${value}`)
    );
  };

  const handleClearLog = (): void => {
    setLog([]);
  };

  const logValue = useMemo(() => log.join('\n'), [log]);

  return (
    <FormProvider {...methods}>
      <FormStyled onSubmit={handleSubmit(handleFormSubmit)}>
        <Input />
        <SubmitButton />
        <Button onClick={handleClearLog}>clear log</Button>
      </FormStyled>

      <CommandLog value={logValue} />
    </FormProvider>
  );
};

export default Form;
