import QtQuick 2.0
import QtQuick.Layouts 1.1
import QtQuick.Controls 1.2
import QtQuick.Controls.Styles 1.2
import com.galileosky.cppclasses 1.0
import "CheckVarType.js" as CheckVarType

Item {
    id: varsPanel
    objectName: "varsPanel"

    property alias panelTitle: panelButton.text
    property string varCategory: "__"

    property var model: globalVarsProxyModel

    signal varNameChanged(string oldName, string newName)

    function acceptChanges(row, role, oldValue, newValue) {
        console.debug("editor.onAccepted:",
                      "row=", row,
                      "role=", role,
                      ":", oldValue, "->", newValue)
        if (newValue === oldValue)
            return;


        var valueCheck = role === "value" ? newValue : model.get(row, "value");

        if (role === "name")
            varNameChanged(oldValue, newValue)

        var varNewRow = model.set(row, role, newValue);

        var varType = "string";
        if (CheckVarType.isFloat(valueCheck)){
            varType = "float";
        }else if (CheckVarType.isNumber(valueCheck)){
            varType = "int";
        }//иначе останется string

        model.set(row, "type", varType);

        // change current row according to resorted model
        if (varNewRow !== row) {
            varTableView.selection.deselect(row);
            varTableView.selection.select(varNewRow);
            varTableView.currentRow = varNewRow;
        }
    }

    height: column.height

    Component.onCompleted: {
        varsPanel.varNameChanged.connect(diagramContainer.diagramField.varNameChanged)
    }

    Component {
        id: varDelegate
        FocusScope {
            Rectangle {
                anchors.fill: parent
                anchors.leftMargin: 6
                border.color: colorSelected
                border.width: styleData.selected ? (loaderEditor.item.activeFocus ? 1 : 0) : 0
                color: "transparent"                

                Text {
                    anchors.fill: parent
                    anchors.leftMargin: 2
                    elide: styleData.elideMode
                    verticalAlignment: Text.AlignVCenter
                    text: styleData.value
                    font.family: "Verdana"
                    font.pointSize: 8
                    font.letterSpacing: 0.5
                    color: (styleData.role === "value") ? (CheckVarType.isFloat(text) ? "Blue" : CheckVarType.isNumber(text) ? colorTextPanel : "green") : colorTextPanel
                    visible: !styleData.selected
                }
            }
            Loader {
                id: loaderEditor

                anchors.fill: parent
                anchors.leftMargin: 6

                sourceComponent: styleData.selected ? editor : null
                Component {
                    id: editor
                    TextInput {
                        id: textInput

                        anchors.fill: parent
                        verticalAlignment: Text.AlignVCenter
                        font.family: "Verdana"
                        font.pointSize: 8
                        font.letterSpacing: 0.5
                        font.bold: styleData.selected
                        text: styleData.value
                        color: (styleData.role === "value") ? (CheckVarType.isFloat(text) ? "Blue" : CheckVarType.isNumber(text) ? colorTextPanel : "green") : colorTextPanel
                        clip: true
                        focus: true

                        onActiveFocusChanged: {
                            console.debug("onActiveFocusChanged:");
                            if (activeFocus === false) {
                                acceptChanges(styleData.row, styleData.role, styleData.value, text)
                            }
                            else
                            {
                                diagramContainer.diagramField.selectionChanged(
                                            diagramContainer.diagramField, undefined, true)
                                focus = true
                            }                            
                        }

                        Keys.onEscapePressed: {
                            console.debug("onEscapePressed:");
                            focus = false;
                            event.accepted = true;
                        }
                        onAccepted: {       // process ENTER
                            console.debug("onAccepted:");
                            focus = false;                            
                        }
                    }
                }
            }
        }
    }

    Component {
        id: typeDelegate
        Item {
            Image {
                id: deleteButton
                anchors.centerIn: parent
                source: "images/icon_del.svg"
                sourceSize.width: imageHeight
                sourceSize.height: imageHeight
                visible: styleData.selected

                MouseArea {
                    anchors.fill: parent

                    onClicked: {
                            var varName = varsPanel.model.get(styleData.row, "name")
                        console.debug("deleteButotn.onClicked:", styleData.row, varName)
                        varNameChanged(varName, "")
                            varsPanel.model.removeRows(styleData.row)
                        scriptGenerator.pushState(diagramContainer)
                    }
                }
            }
        }
    }

    Column {
        id: column
        width: parent.width

        Rectangle {

            anchors.left: parent.left
            anchors.right: parent.right
            height: panelCaptionHeight
            color: colorGroup
            border.width: 1
            border.color: colorScrollBorder

            RowLayout {
                anchors.fill: parent
                anchors.rightMargin: 12

                ActionToolButton {
                    id: panelButton

                    Layout.fillWidth: true
                    Layout.fillHeight: true
                }
                Button {
                    id: addButton
                    objectName: "addButton"

                    implicitWidth: Math.round(24 * kDPI)
                    implicitHeight: Math.round(24 * kDPI)
                    text: qsTr("+")

                    style: ButtonStyle {
                        background:  Item {
                            anchors.fill: parent
                            Rectangle {
                                id: shadowRect
                                x: control.pressed ? 0 : 1
                                y: control.pressed ? 0 : 1
                                implicitWidth:  parent.width-1
                                implicitHeight: parent.height-1
                                radius: Math.round(6 * kDPI)
                                color: colorToolbarButtonShadow
                            }
                            Rectangle {
                                id: foreRect
                                x: control.pressed ? 1 : 0
                                y: control.pressed ? 1 : 0
                                implicitWidth:  parent.width-1
                                implicitHeight: parent.height-1
                                radius: Math.round(6 * kDPI)
                                color: colorWhitespace
                            }
                        }
                        label: Text {
                            id: name

                            x: control.pressed ? 1 : 0
                            y: control.pressed ? -1 : -2
                            verticalAlignment: Text.AlignVCenter
                            horizontalAlignment: Text.AlignHCenter

                            font.family: "Verdana"
                            font.pointSize: 14
                            font.letterSpacing: 1.0
                            text: control.text
                        }
                    }

                    onClicked: {
                        console.debug(varsPanel.panelTitle, "onClicked:")
                        focus = true;   // steal focus from varTable to accept changes
                        model.addVar(varsPanel.varCategory)
                        scriptGenerator.pushState(diagramContainer)

                        if (panelButton.checked === false)
                            panelButton.checked = true
                    }
                }
            }
        }

        TableView {
            id: varTableView
            objectName: "varTableView"

            property real rowHeight: Math.round(18 * kDPI)

            anchors.left: parent.left
            anchors.right: parent.right
            height: panelButton.checked ? (rowHeight * rowCount) : 0
            headerVisible: false
            alternatingRowColors: false
            frameVisible: false
            backgroundVisible: false
            model: varsPanel.model
            focus: true

            rowDelegate: Rectangle {
                height: varTableView.rowHeight
                color: "transparent"
                border.width: 1
                border.color: colorScrollBorder
            }
            style: TableViewStyle {
                scrollBarBackground: Item {}
                handle: Item {}
                decrementControl: Item {}
                incrementControl: Item {}
            }

            TableViewColumn {
                id: typeName
                role: "name"
                title: qsTr("VarName")
                width: (varTableView.width - typeColumn.width) / 2

                delegate: varDelegate
            }
            TableViewColumn {
                role: "value"
                title: qsTr("Value")
                width: varTableView.width - typeName.width - typeColumn.width - 17

                delegate: varDelegate
            }
            TableViewColumn {
                id: typeColumn
                role: "type"
                title: qsTr("Type")
                width: Math.round(16 * kDPI)    // deleteButton.width

                delegate: typeDelegate
            }

            onClicked: console.debug(varsPanel.title, "onClicked:", row)
            onDoubleClicked: console.debug(varsPanel.title, "onDoubleclicked:", row)

            Behavior on height {
                NumberAnimation {
                    easing.type: Easing.OutQuad
                }
            }
        }
    }
}
