import QtQuick 2.2
import QtQuick.Controls 1.2
import QtQml.Models 2.1
import com.galileosky.cppclasses 1.0

Item {
    id: pageGridItem
    objectName: "pageGridItem"

    property alias pageView: pageGridView
    property alias pageIndex: pageGridView.currentIndex
    property alias pageCount: pageGridView.count

    property ListModel pageModel: ListModel {}
    property int defaultCellWidth: Math.round(150 * kDPI)

    clip: true

    Component.onCompleted: {
        for (var i = 0; i < 4; ++i)
            pageModel.append({"pageName" : "<DEFAULT>"});
        scriptGenerator.pushState(diagramContainer)
    }

    onWidthChanged: setPageViewWidth()

    function dump() {
        console.debug("dump", objectName, "pageModel:");
        for (var i = 0; i < pageModel.count; ++i) {
            console.debug(i, ":", pageModel.get(i).pageName);
        }
        scriptGenerator.scriptModifiedFromQML();
    }

    function getCurrentPageName() {
        return pageView.getPageName(pageIndex);
    }

    function getPageName(index) {
        var pageName = pageModel.get(index).pageName;
        return (pageName === "<DEFAULT>") ? ("Page " + index) : pageName;
    }

    function setPageViewWidth() {
        pageView.width = ((defaultCellWidth * pageModel.count) > (parent.width - addPageButton.width))
                ? (parent.width - addPageButton.width)
                : (defaultCellWidth * pageModel.count)
        pageView.cellWidth = ((defaultCellWidth * pageModel.count) > (parent.width - addPageButton.width))
                ? ((parent.width - addPageButton.width) / pageModel.count)
                : defaultCellWidth
    }

    GridView {
        id: pageGridView
        objectName: "pageGridView"

        width: defaultCellWidth
        height: parent.height

        cellWidth: defaultCellWidth
        cellHeight: parent.height
        flow: GridView.TopToBottom
        interactive: false

        displaced: Transition {
            NumberAnimation { properties: "x,y"; easing.type: Easing.OutQuad }
        }
        model: visualModel

        onCountChanged: setPageViewWidth()

        DelegateModel {
            id: visualModel
            model: pageModel
            delegate: MouseArea {
                id: pageDelegate

                property int visualIndex: DelegateModel.itemsIndex
                property string pageName: model.pageName

                width: pageGridView.cellWidth
                height: pageGridView.cellHeight
                drag.target: delegateItem
                drag.axis: Drag.XAxis
                acceptedButtons: Qt.LeftButton | Qt.RightButton

                Menu {
                    id: pageMenu

                    MenuItem {
                        text: qsTr("Delete page")
                        onTriggered: {
                            pageModel.remove(pageDelegate.visualIndex)
                            diagramField.removePage(pageDelegate.visualIndex);
                            dump();
                        }
                    }
                }

                Item {
                    id: delegateItem
                    anchors.horizontalCenter: parent.horizontalCenter
                    anchors.verticalCenter: parent.verticalCenter
                    width: pageGridView.cellWidth
                    height: pageGridView.cellHeight

                    Drag.active: pageDelegate.drag.active
                    Drag.source: pageDelegate
                    Drag.hotSpot.x: width / 2
                    Drag.hotSpot.y: height / 2
	    
                    Image {
                        id: delegateImageLeft
                        objectName: "tabLeftImage"
                        anchors.left: parent.left
			anchors.bottom: parent.bottom
			anchors.bottomMargin: -1
                        z:10
                        sourceSize.height: pageGridView.cellHeight
			sourceSize.width: parent.width / 5	
                        source: (pageGridView.currentItem === pageDelegate) ? "images/tab_left_on.svg" : "images/tab_left_off.svg"
                    }
                    Image {
                        id: delegateImageRight
                        objectName: "tabRightImage"
                        anchors.right: parent.right
			anchors.bottom: parent.bottom
			anchors.bottomMargin: -1
                        z:10
                        source: (pageGridView.currentItem === pageDelegate) ? "images/tab_right_on.svg" : "images/tab_right_off.svg"
                        sourceSize.height: pageGridView.cellHeight
			sourceSize.width: parent.width / 5	
                    }
                    Image {
                        id: delegateImageCenter
                        objectName: "tabCenterImage"
                        anchors.left: delegateImageLeft.right
                        anchors.right: delegateImageRight.left
                        anchors.bottom: parent.bottom
                        anchors.bottomMargin: -1
                        z:10
                        source: (pageGridView.currentItem === pageDelegate) ? "images/tab_center_on.svg" : "images/tab_center_off.svg"
                        sourceSize.height: pageGridView.cellHeight
                        sourceSize.width: 3*parent.width / 5			

                        Rectangle {
                            id: delegateTextDecorator

                            anchors.fill: delegateText
                            anchors.margins: Math.round(-5 * kDPI)
                            radius: Math.round(10 * kDPI)
                            color: colorEditing
                            visible: delegateText.activeFocus
                        }
                        TextInput {
                            id: delegateText

                            anchors.centerIn: parent
                            font.family: "Verdana"
                            font.bold: true
                            font.pointSize: 8
                            color: activeFocus ? colorTextEditing : colorPageSelectorText

                            text: ((pageDelegate.pageName === "<DEFAULT>")
                                   ? qsTr("Page ") + index
                                   : pageDelegate.pageName)
                            enabled: false

                            function setPageName(name) {
                                if (name === ("Page " + index))
                                    pageModel.set(index, {"pageName" : "<DEFAULT>"})
                                else
                                    pageModel.set(index, {"pageName" : displayText})
                                scriptGenerator.scriptModifiedFromQML();
                            }

                            Keys.onEscapePressed: {
                                setPageName(displayText)
                                enabled = false
                            }

                            onEditingFinished: {
                                setPageName(displayText)
                                enabled = false
                            }
                        }
                    }

                    states: [
                        State {
                            when: delegateItem.Drag.active
                            ParentChange {
                                target: delegateItem
                                parent: pageGridView
                            }
                            AnchorChanges {
                                target: delegateItem;
                                anchors.horizontalCenter: undefined;
                                anchors.verticalCenter: undefined
                            }
                        }
                    ]
                }

                onClicked: {
                    if (mouse.button === Qt.RightButton) {
                        pageMenu.popup();
                    }
                    else if (mouse.button === Qt.LeftButton) {
                        delegateItem.forceActiveFocus();
                        pageView.currentIndex = visualIndex;
                    }
                }
                onDoubleClicked: {
                    if (delegateText.activeFocus === false) {
                        delegateText.enabled = true;
                        delegateText.forceActiveFocus();
                        delegateText.selectAll();
                    } else {
                        delegateText.enabled = false;
                    }
                }

                DropArea {
                    anchors { fill: parent; }

                    onEntered: {
                        if (drag.source.visualIndex === pageDelegate.visualIndex)
                            return;

                        console.debug("DropArea.onEntered:",
                                      drag.source.visualIndex, "->",
                                      pageDelegate.visualIndex);
                        pageModel.move(drag.source.visualIndex,
                                       pageDelegate.visualIndex, 1);
                        diagramField.movePage(drag.source.visualIndex,
                                              pageDelegate.visualIndex);
                    }
                }
            }
        }
    }
    Item {
        id: addPageButton

        anchors.left: pageGridView.right
        anchors.top: pageGridView.top
        anchors.topMargin: 1
        anchors.bottom: pageGridView.bottom
        anchors.bottomMargin: -1
        width: defaultCellWidth / 3

        visible: !pageGridView.dragging

        Image {
            id: addPageImageLeft
            objectName: "tabLeftImage"
            anchors.left: parent.left
            anchors.top: parent.top
            anchors.bottom: parent.bottom
            source: "images/tab_left_off.svg"
            sourceSize.height: pageGridView.cellHeight
	    sourceSize.width: parent.width / 3
        }
        Image {
            id: addPageImageRight
            objectName: "tabRightImage"
            anchors.right: parent.right
            anchors.top: parent.top
            anchors.bottom: parent.bottom
            source: "images/tab_right_off.svg"
            sourceSize.height: pageGridView.cellHeight
	    sourceSize.width: parent.width / 3	
        }
        Image {
            id: addPageImageCenter
            objectName: "tabCenterImage"
            anchors.left: addPageImageLeft.right
            anchors.right: addPageImageRight.left
            anchors.top: parent.top
            anchors.bottom: parent.bottom
            source: "images/tab_center_off.svg"
            sourceSize.height: pageGridView.cellHeight
	    sourceSize.width: parent.width / 3

            Text {
                anchors.centerIn: parent
                font.family: "Verdana"
                font.bold: true
                font.pointSize: 13
                text: "+"
            }
            MouseArea {
                anchors.fill: parent

                onClicked: {
                    pageModel.append({"pageName" : "<DEFAULT>"});
                    dump();
                }
            }
        }
    }
}
