import { nanoid } from 'nanoid';
import { ICommanderProps, TCommandSend } from '../types/commander';
import { IQueueItem } from '../types/queue';
import Api from './api';
import Logger from './logger';
import Queue from './queue';

class Commander {
  /**
   * Api wrap
   */
  private _api: Api;

  /**
   * Command queue
   */
  private _queue: Queue;

  /**
   * Request timeout in ms
   */
  private _timeout = 10000;

  /**
   * Amount of retries on request error
   */
  private _retries = 3;

  private _isInitialized = false;

  constructor(props?: Partial<ICommanderProps>) {
    this._queue = new Queue();
    this._api = new Api(this._queue);

    this._timeout = props?.timeout ?? this._timeout;
    this._retries = props?.retries ?? this._retries;

    this._init();
  }

  private _init = (): void => {
    if (this._isInitialized) {
      return;
    }

    this._isInitialized = true;
  };

  public send: TCommandSend = (args) => {
    const id = nanoid();
    const timeout = args.timeout ?? this._timeout;
    const retries = args.retries ?? this._retries;

    Logger.log(`submit command: ${args.request}, id: ${id}`);

    return new Promise<string>((resolve, reject) => {
      const command = async () => {
        try {
          const response = await this._api.send(id, args.request, timeout);
          resolve(response);

          this._queue.next();
        } catch {
          this._queue.retry();
        }
      };

      const queueItem: IQueueItem = {
        id,
        command,
        retries,
        reject,
      };

      this._queue.add(queueItem);
    });
  };

  public destroy = (): void => {
    if (!this._isInitialized) {
      return;
    }

    this._isInitialized = false;
  };
}

export default Commander;
