import QtQuick 2.2
import QtQuick.Controls 1.2
import QtQuick.Controls.Styles 1.2
import QtQuick.Layouts 1.1
import com.galileosky.cppclasses 1.0

Rectangle {
    id: scriptDialog
    objectName: "scriptDialog"

    property var diagramItem: null
    property var parametersModel: null

    property string scriptName

    property int paramaterItemHeight: Math.round(20 * kDPI)
    property real centerPointX: 0.0
    property real centerPointY: 0.0
    property real scaleFactorX: 0.0
    property real scaleFactorY: 0.0
    property string compileResult

    property alias dialogTitle: dialogTitleText.text
    property var scriptTemplate:   "// New script template.\n" +
                                    "\n" +
                                    "// Write script description here.\n" +
                                    "\n" +
                                    "main()\n" +
                                    "{\n" +
                                    "    // Get terminal time\n" +
                                    "    new time = GetVar(UNIX_TIME)\n" +
                                    "\n" +
                                    "    // Get terminal status\n" +
                                    "    new status = GetVar(STATUS)\n" +
                                    "\n" +
                                    "    // Post diagnostic message\n" +
                                    "    Diagnostics(\"Terminal time: %d. Terminal status: %d\", time, status);\n" +
                                    "}";

    width: Math.round(600 * kDPI)
    height: Math.round(400 * kDPI)
    radius: Math.round(10 * kDPI)
    z: zParameterDialog
    color: colorDialogBackground
    border.width: 1
    border.color: colorDialogBorder
    visible: false

    function dump() {
        console.debug("---- dump start:", objectName);
        console.debug(dialogTitle);
        var i;
        console.debug("parametersModel:", parametersModel);
        for (i = 0; i < parametersModel.count; ++i)
            console.debug(parametersModel.get(i).paramName,
                          parametersModel.get(i).paramType,
                          parametersModel.get(i).varName);
        console.debug("---- dump finish:", objectName);
    }

    function showItem(item) {
        diagramItem = item
        if (diagramItem.destroying)
            diagramItem.destroying.connect(function(){
                accept();
                scriptDialog.destroy();
            });
        dialogTitle = qsTranslate(item.actionName, item.actionName)
        parametersModel = item.parametersModel;
        scriptName = parametersModel.get(parametersModel.paramIndex).varName;

        dump();

        // center dialog to item
        var itemCenter = diagramContainer.mapFromItem(item, item.width/2, item.height/2);
        centerPointX = itemCenter.x;
        centerPointY = itemCenter.y;
        // set size to item and show,...
        scaleFactorX = diagramContainer.diagramField.itemWidth / width;
        scaleFactorY = diagramContainer.diagramField.itemHeight / height;
        visible = true;
        // ...than animate to normal size
        centerPointX = parent.width/2
        centerPointY = parent.height/2
        scaleFactorX = 1.0;
        scaleFactorY = 1.0;
    }

    function selectionChanged(item, mouseEvent, itemWasSelected) {
        console.debug(objectName, "selectionChanged()", item)
        if ((visible === false) || (item === scriptDialog))
            return;

        accept();
        dump();
    }

    //! Save script contents: modifying name + modifying contents
    function saveScript() {
        console.debug(arguments.callee.name, scriptName, textEditor.text);
        if (textEditor.modified === true) {
            scriptVarsProxyModel.set(
                        scriptVarsProxyModel.getVarId(scriptName, categoryConst),
                        "value", textEditor.text);
            textEditor.modified = false;
        }
    }

    function loadScript() {
        if (scriptName.length === 0) {
            textEditor.text = scriptTemplate;
            scriptNameItem.forceActiveFocus();
        }
        else {
            textEditor.text =
                scriptVarsProxyModel.get(
                        scriptVarsProxyModel.getVarId(scriptName, categoryConst),
                        "value");
            textEditor.forceActiveFocus();
        }
        compileText.visible = false;
        textEditor.modified = false;
        console.debug(arguments.callee.name, scriptName, textEditor.text);
    }

    function accept() {
        console.debug(objectName, arguments.callee.name);
        if (parametersModel)
            saveScript()

        visible = false;
    }

    Component.onCompleted: {
        diagramField.selectionChanged.connect(scriptDialog.selectionChanged)
        console.debug(objectName, "comleted")
    }
    Component.onDestruction: {
        diagramField.selectionChanged.disconnect(scriptDialog.selectionChanged)
        console.debug(objectName, "destructed")
    }

    onScriptNameChanged: {
        loadScript();
    }

    MouseArea {
        anchors.fill: parent
        drag.target: parent
        acceptedButtons: (Qt.LeftButton | Qt.RightButton)
        onWheel: { wheel.accepted = true; }
    }

    ColumnLayout {
        anchors.fill: parent
        anchors.margins: parameterItemMargins

        Item {
            Layout.fillWidth: true
            height: parameterItemHeight
            RowLayout {
                id: titleLayout
                anchors.fill: parent

                Text {
                    id: dialogTitleText

                    width: paintedWidth + 4
                    Layout.fillHeight: true
                    verticalAlignment: Text.AlignVCenter
                    font.bold: true
                }
                VarNameItem {
                    id: scriptNameItem
                    objectName: "scriptNameItem"

                    Layout.fillWidth: true
                    Layout.fillHeight: true
                    paramModel: parametersModel
                    paramIndex: 0

                    onVarNameChanged: {
                        scriptName = varName;
                    }
                }
                Image {
                    id: closeRect

                    source: closeRectArea.containsMouse?"images/icon_close.svg":"images/icon_gray_close.svg"
                    sourceSize.width: Math.round(18 * kDPI)
                    sourceSize.height: Math.round(18 * kDPI)
                    smooth: false

                    MouseArea {
                        id: closeRectArea
                        objectName: "closeRectArea"
                        anchors.fill: parent
                        cursorShape: Qt.PointingHandCursor
                        hoverEnabled: true
                        onClicked: {
                            console.debug("scriptDialog. Close clicked")
                            accept()
                        }
                    }
                }
            }
        }

        Item {
            id: editRect

            Layout.fillWidth: true
            Layout.fillHeight: true

            Rectangle {
                anchors.fill: parent
                anchors.margins: parameterItemMargins/2
                border.width: textEditor.activeFocus ? 2 : 1
                border.color: ((textEditor.length === 0) || textEditor.readOnly)
                              ? colorError
                              : (textEditor.modified === true) ? colorModified : colorOk
                TextArea {
                    id:textEditor

                    property bool modified: false

                    anchors.fill: parent
                    anchors.margins: 1  // to make visible parent border
                    activeFocusOnTab: true
                    text: documentHandler.text

                    enabled: scriptName

                    style: TextAreaStyle {
                        scrollBarBackground: Rectangle {
                            implicitWidth: Math.round(8 * kDPI)
                            border.width: 2
                            border.color: colorScrollBorder
                            color: colorScrollBackground
                        }
                        handle: Rectangle {
                            implicitWidth: Math.round(8 * kDPI)
                            radius: Math.round(4 * kDPI)
                            color: colorScrollHandle
                        }

                        decrementControl: Component { Item {} }
                        incrementControl: Component { Item {} }

                        frame: Item {}

                        font {
                            family: "Consolas";
                            pointSize: 8;
                        }
                        textColor: colorTextPanel
                    }

                    Component.onCompleted: {
                        textEditor.modified = false;
                    }
                    // \NOTE: tabChangesFocus: false - doesn't work
                    Keys.onTabPressed: {
                        event.accepted = true;
                        insert(cursorPosition, '    ');
                    }

                    onTextChanged: {
                        if ((focus === true) && (textEditor.modified === false)) {
                            console.debug("text was modified")
                            textEditor.modified = true;
                        }
                    }
                    onActiveFocusChanged: {
                        if (activeFocus === false)
                            saveScript()
                    }
                    Keys.onEscapePressed: accept()

                    DocumentHandler {
                        id: documentHandler

                        target: textEditor
                        cursorPosition: textEditor.cursorPosition
                        selectionStart: textEditor.selectionStart
                        selectionEnd: textEditor.selectionEnd

                        Component.onCompleted: {
                            text = scriptTemplate;
                        }

                        onTextChanged: {
                            textEditor.update()
                        }
                    }
                }

                MouseArea {
                    anchors.fill: parent
                    hoverEnabled: true
                    enabled: !textEditor.enabled
                    onMouseYChanged: {
                         // что бы применилось имя скрипта и разблочился textEditor
                        if ( scriptNameItem.localVarName.length!=0)
                            textEditor.forceActiveFocus();
                    }
                }
            }
        }

        Button {
            id: compileButton
            objectName: "compileButton"

            property int result: 0

            text: qsTr("Compile script")

            onClicked: {
                forceActiveFocus();
                compileText.visible = true;
                console.debug(objectName, "onClicked:")
                result = scriptGenerator.compileScript(scriptName)
                if (result === 0) {
                    compileResult = qsTr("Script has been compiled successfully");
                }
                else {
                    compileResult = qsTr("Compile error %1 : %2")
                            .arg(result)
                            .arg(scriptGenerator.getCompiledScriptError());
                }
            }
        }
        Text {
            id: compileText
            objectName: "compileResult"

            text: ((textEditor.modified === true) ? "* " : "")
                  + compileResult
        }
    }

    Image {
        source:  "images/icon_gray_resize.svg"
        sourceSize.width: Math.round(20 * kDPI)
        sourceSize.height: Math.round(20 * kDPI)
        anchors.right: parent.right
        anchors.bottom: parent.bottom
        clip: true
        MouseArea {
            anchors.fill: parent
            cursorShape: Qt.SizeFDiagCursor
            hoverEnabled: true

            drag{ target: parent; axis: Drag.XAndYAxis }
            onMouseYChanged: {
                if(drag.active){
                    var dy = mouseY
                    if(scriptDialog.height + dy >= 150) {
                        scriptDialog.y += dy / 2
                        scriptDialog.height += dy
                    }
                }
            }
            onMouseXChanged: {
                if(drag.active){
                    var dx = mouseX
                    if(scriptDialog.width + dx >= 200) {
                        scriptDialog.x += dx / 2
                        scriptDialog.width += dx
                    }
                }
            }
        }
    }

    transform: [
        Scale {
            origin.x: width/2
            origin.y: height/2
            xScale: scaleFactorX
            yScale: scaleFactorY
        },
        Translate {
            x: centerPointX - width/2
            y: centerPointY - height/2
        }
    ]

    Behavior on scaleFactorX {
        enabled: visible
        NumberAnimation {
            easing.type: Easing.Linear
        }
    }
    Behavior on scaleFactorY {
        enabled: visible
        NumberAnimation {
            easing.type: Easing.Linear
        }
    }
    Behavior on centerPointX {
        enabled: visible
        NumberAnimation {
            easing.type: Easing.Linear
        }
    }
    Behavior on centerPointY {
        enabled: visible
        NumberAnimation {
            easing.type: Easing.Linear
        }
    }
}
