import QtQuick 2.2
import QtQuick.Controls 1.2
import QtQuick.Controls.Styles 1.2
import QtQuick.Layouts 1.1
import QtQuick.Dialogs 1.2
import Qt.labs.platform 1.0
import QtCharts 2.2
import com.galileosky.cppclasses 1.0

import "auxiliary.js" as Code
import "averageCalculation.js" as Avg

Rectangle {
    id: fuelTarFilterDialog
    objectName: "fuelTarFilterDialog"

    property var diagramItem: null
    property var parametersModel: ListModel{
        ListElement{ // пустой элемент для избежания ошибки при создании таймера.
            VarName: ""
        }
    }

    property int header_height: 20
    property int body_height: 440
    property int footer_height: 0

    property int margin_offset : 5
    property int row_height: 20

    // Свойства для изменения положения диалога и его размера
    property real centerPointX: 0.0
    property real centerPointY: 0.0
    property real scaleFactorX: 0.0
    property real scaleFactorY: 0.0

    property var saveFileDialog: null;
    property int loaded_value_index: 0;
    property int loaded_value: 0;
    property int average_value: 0;
    property int pass_time: 0;
    property int measurment_count: 0;

    property var current_cell: null
    property int current_cell_index: 0

    property string table_std_color: colorEditing
    property string table_y_std_color: colorEditing
    property string table_active_color: colorSelected
    property string table_sustained_value_color: "lightgreen"
    property string table_gather_value_color: "gold"
    property string buttons_color: "white"
    property string std_transparent: "transparent"
    property string std_body: "white"
    property string std_popup: colorTextActionFinish
    property string mes_color: "white"
    property string second_title: ""


    property int fuel_temp_container_height: 60
    property int fuel_temp_container_width: 300;

    property int cal_table_top_offset: fuel_temp_container_height

    property int tar_table_model_parameter_start_pos: 5

    property int recieved_temperature: 10; // градус Цельсия *10. Температура получаемая от прибора
    // Температура после нажатия кнопки принять температуру.  Устанавливается в модель данных parametersModel
    property int accepted_temperature: 0; // градус Цельсия *10.

    property int font_height: 15

    // Индес для определения какой из 2-х параметров топлив/температура опрашивается в данный момент.
    property int current_param_index_pos: 0;
    property int current_param_index: 0;

    // Сохранение значения для оси x
    property int origin_axis_x_value: 0;

    property var fuel_type_object: null

    property ListModel table_model: ListModel{
        ListElement{
            axis_x: qsTr("FLS Value"); // Значение датчика
            axis_y: qsTr("Liters"); // Приведенное к единицам измерения
            is_start: "true"
        }

        onDataChanged: {
            console.debug("***** TABLE_MODEL CHANGED");
            changeLineSeries(table_model);
        }

        onCountChanged: {
            console.debug("***** TABLE_MODEL COUNT CHANGED");
            changeLineSeries(table_model);
        }

        onRowsInserted: {
            console.debug("***** TABLE_MODEL ROWS INDESRTED");
            changeLineSeries(table_model);
        }
    }

    width: 690
    height: header_height*2 + body_height + footer_height + margin_offset*2
    radius: Math.round(10 * kDPI)

    color: colorDialogBackground
    border.width: 1
    border.color: colorDialogBorder

    z: zParameterDialog

    // Задает возможность перетаскивать диалоговое окно
    MouseArea {
        anchors.fill: parent
        drag.target: parent
        acceptedButtons: (Qt.LeftButton | Qt.RightButton)
        onPressed: {
            console.debug("z index:", fuelTarFilterDialog.z);
            thisDialogPressed(diagramItem)
        }
    }

    // Это самая важная штука для получения данных от прибора
    // Этот таймер дергает плюсовый объект, отвечающий за получение данных с прибора
    // Запуск/Остановка таймера:
    // Запуск - функция tryToStartTimer() - там идет проверка установлена ли переменная от которой будут полкчены данные. Вызывается в showItem()
    // Остановка - функция accept() - она вызывается при сокрытии окна диалгоа.
    Timer {
        id: load_val_timer
        interval: 1000;
        running: false
        repeat: true
        onTriggered: {
            ++pass_time;
            // попытка установить полученное значение
            setupRecievedValue();
            second_title = diagramItem.getConnectedVarName();
            if (null !== current_cell) {
                setupCellValue(current_cell);
            }
        }
    }

    // Функция установки диалога на первый план. Обработчик сигнала
    function setupZ(item){
        if (diagramItem === item) {
            z = zParameterDialog + 1;
            diagramItem.itemSelected(diagramItem, undefined);
        } else {
            z = zParameterDialog;
        }
    }
    // Сигнал отправляемый при нажатии на диалоговое окно
    signal thisDialogPressed(var item);

    property int index_row_to_delete: 0
    Item{
       id: signals_handler
       visible: false;

       signal delete_table_row();

       onDelete_table_row: {
           console.log("signals_handler: delete_table_row emitted");
           removeCalibrationData(index_row_to_delete);
       }
    }

    // Здесь создается FocusScope, чтобы отслеживать получение фокуса элементами управления.
    // Такой подход позволяет обеспечить определение действия пользователя с диалоговым окном и отследить смену окон
    // Все равноне так удобно потому что мы ориентируемся на ActiveFocus, получение которого, в большинстве случаев,
    // необходимо прописывать в описани элемента самостоятельно. Даже Button автоматически его не захватывает.
    // А вот ComboBox и TextInput это делают.
    FocusScope {
        id: scope

        anchors.fill: parent
        //FocusScope needs to bind to visual properties of the children. Это из документации
        property alias color: scope_area.color

        onActiveFocusChanged: {
            if (scope.activeFocus){
                thisDialogPressed(diagramItem);
            }
        }

        Rectangle {
            id: scope_area
            anchors.fill: parent
            color: "transparent"
            Rectangle {
                id: main_container
                anchors.fill: parent
                anchors.leftMargin: margin_offset
                anchors.rightMargin: margin_offset
                anchors.topMargin: margin_offset
                anchors.bottomMargin: margin_offset

                Rectangle{
                    id: header
                    height: header_height;
                    width: parent.width
                    color: std_body
                    anchors.top: parent.top

                    Text{
                        font.bold: true
                        font.pixelSize: 12
                        text: qsTr("Fuel Tar Filter Dialog");
                    }

                    Rectangle {
                        id: closeRect
                        anchors.right: parent.right;
                        anchors.top: parent.top;
                        anchors.topMargin: 0
                        anchors.rightMargin: 0
                        width: Math.round(18 * kDPI)
                        height: Math.round(18 * kDPI)

                        Image {
                            id: closeImage

                            source: closeRectArea.containsMouse?"images/icon_close.svg":"images/icon_gray_close.svg"
                            sourceSize.width: Math.round(18 * kDPI)
                            sourceSize.height: Math.round(18 * kDPI)
                            smooth: false

                            MouseArea {
                                id: closeRectArea
                                objectName: "closeRectArea"
                                anchors.fill: parent
                                cursorShape: Qt.PointingHandCursor
                                hoverEnabled: true
                                onClicked: {
                                    console.debug("fuelTarFilterDialog. Close clicked")
                                    if (diagramItem) {
                                        accept();
                                        diagramItem.is_dialog_created = false;
                                        diagramItem.notWantAnymore();
                                    }
                                    forceActiveFocus();
                                    fuelTarFilterDialog.destroy();
                                }
                            }
                        }
                    }
                }

                Rectangle{
                    id: underheader
                    height: header_height;
                    width: parent.width
                    color: std_body
                    anchors.top: header.bottom

                    Text{
                        anchors.left: parent.left;
                        anchors.leftMargin: 20;
                        anchors.top: parent.top
                        anchors.topMargin: 3;
                        font.pixelSize: 10
                        text: second_title;
                    }
                }

                Rectangle{
                    id: body
                    height: body_height;
                    width: parent.width
                    color: std_body

                    anchors.top: underheader.bottom

                    Rectangle {
                        id: acc_temp_lbl_rect
                        width: accepted_temp_value_txt.width
                        height: 60

                        anchors.top: parent.top;
                        anchors.topMargin: margin_offset

                        anchors.right: parent.right
                        anchors.rightMargin: margin_offset*4

                        Text {
                            id: accepted_temp_value_txt
//                            anchors.centerIn: parent;
                            anchors.top: parent.top;
                            font.pixelSize: font_height
                            text: qsTr("Accepted temperature Value\n for calibration:").arg(accepted_temperature);
                        }
                        Rectangle {
                            width: acc_temp_txt.width + 6
                            height: row_height
                            border.width: 1
                            border.color: colorDialogBorder
                            anchors.top: accepted_temp_value_txt.bottom
                            anchors.right: parent.right
                            anchors.rightMargin: 50
                            TextInput {
                                id: acc_temp_txt
                                text: qsTr(" %1").arg(accepted_temperature);
                                font.pixelSize: font_height
                                validator: RegExpValidator { regExp:/[0-9]+/ }
                                onAccepted: {
                                    accepted_temperature = parseInt(text);
                                    focus = false;
                                }
                                onEditingFinished: {
                                    accepted_temperature = parseInt(text);
                                    focus = false;
                                }
                                onFocusChanged: {
                                    if (focus) {
                                        selectAll();
                                    }
                                }
                            }
                            Text {
                                id: celsium
                                text: " \u00B0C";
                                font.pixelSize: font_height
                                anchors.left: acc_temp_txt.right
                                anchors.leftMargin: margin_offset
                            }
                        }
                    }

                    Rectangle{
                        id: fuel_temp;

                        width: fuel_temp_container_width;
                        height: fuel_temp_container_height;

                        color: std_body

                        anchors.top: parent.top;
                        anchors.topMargin: margin_offset;
                        anchors.left: parent.left;
                        anchors.leftMargin: margin_offset*2;

                        Rectangle{
                            id: fuel_rect
                            width: parent.width
                            height: row_height;

                            anchors.top: parent.top;
                            anchors.topMargin: margin_offset;
                            Row {
                                width: parent.width
                                height: parent.height;
                                Text{
                                    id: fuel_txt_lable
                                    width: 100;
                                    height: parent.height;
                                    text: qsTr("fuel type:")
                                    font.pixelSize: font_height
                                }
                                // Здесь выбор переменной с параметром
                                VarNameItem{
                                    width: 200
                                    height: row_height;
                                    paramModel: parametersModel
                                    paramIndex: 2
                                    is_text_allowed: true
                                    // Тут хитрая штука, чтобы при потере фокуса устанавливалось значение
                                    // Без этого работает только Enter|Return
                                    comboBox.onFocusChanged: {
                                        if (!focus) {
                                            paramModel.set(paramIndex, {"varName": comboBox.editText});
                                        }
                                    }
                                }
                            }
                        }
                        Rectangle{
                            width: parent.width
                            height: row_height;

                            anchors.top: fuel_rect.bottom;
                            anchors.topMargin: margin_offset;
                            Row {
                                width: parent.width
                                height: row_height;
                                Text{
                                    id: temp_txt_lable
                                    width: 100;
                                    height: parent.height;
                                    text: qsTr("temperature:")
                                    font.pixelSize: font_height
                                }

                                Text{
                                    id: current_temp_value_txt
                                    width: 100;
                                    height: parent.height;
                                    text: "%1 \u00B0C".arg(recieved_temperature);
                                    font.pixelSize: font_height
                                }

                                Text {
                                    id: accept_temperature_btt_txt
                                    text: qsTr("Accept temperature");
                                    anchors.centerIn: parent;
                                    visible: false
                                }
                                Button {
                                    id: accept_temperature_rect_button

                                    text: accept_temperature_btt_txt.text
                                    style: ButtonStyle {
                                        background: Rectangle {
                                            implicitWidth: accept_temperature_btt_txt.width + 20
                                            implicitHeight: row_height
                                            radius: row_height/4
                                            border.width: 1
                                            border.color: "#888"
                                            gradient: Gradient {
                                                GradientStop { position: 0 ; color: control.pressed ? "#ccc" : "#eee" }
                                                GradientStop { position: 1 ; color: control.pressed ? "#aaa" : "#ccc" }
                                            }
                                        }
                                    }
                                    onClicked: {
                                        forceActiveFocus();
                                        accepted_temperature = recieved_temperature;
                                    }
                                }


                            }
                        }
                    }

                    Text {
                        id: add_btt_txt
                        text: qsTr("Accept and add value");
                        anchors.centerIn: parent;
                        visible: false
                    }

                    Button {
                        id: add_slot_table_button
                        anchors.top: calibration_table_container.bottom
                        anchors.topMargin: margin_offset
                        anchors. left: calibration_table_container.left

                        text: add_btt_txt.text;
                        style: ButtonStyle {
                            background: Rectangle {
                                implicitWidth: add_btt_txt.width + 20
                                implicitHeight: row_height
                                radius: row_height/4
                                border.width: 1
                                border.color: "#888"
                                gradient: Gradient {
                                    GradientStop { position: 0 ; color: control.pressed ? "#ccc" : "#eee" }
                                    GradientStop { position: 1 ; color: control.pressed ? "#aaa" : "#ccc" }
                                }
                            }
                        }
                        onClicked: {
                            forceActiveFocus();
                            // Мы определяем текущую выделенную клетку, точнее это TextEdit
                            // Теперь мы должны принять ее текущее значение в таблицу
                            if (null !== current_cell){
                                acceptXValue(current_cell_index, current_cell.text);
                            }
                            // Добавляем новый элемент в таблицу
                            addNewCalibratinData();
                            // Теперь фокус нужно переключить на новый элемент таблицы
                            setupCurrentCell(getTableLastXItem(), lv.count - 1);
                            setupCurrentCellActive();
                        }
                    }

                    Rectangle{
                        id: calibration_table_container
                        width: 240
                        height: 300

                        anchors.top: parent.top
                        anchors.topMargin: margin_offset*2 + cal_table_top_offset

                        anchors.left: parent.left
                        anchors.leftMargin: margin_offset*2

                        color: std_body
                        border.width: 1
                        border.color: colorDialogBorder


                        Rectangle{
                            id: calibration_table

                            property int cell_width: 110
                            property int rem_rect_width: 20
                            property int scroller_width: 10

                            width: (cell_width*2) + rem_rect_width + scroller_width;
                            height: parent.height

                            anchors.top: parent.top
                            anchors.left: parent.left

                            color: std_body
                            border.width: 1
                            border.color: colorDialogBorder

                            Loader {
                                id: popup_cloud
                            }

                            ScrollView {
                                anchors.fill: parent
                                flickableItem.interactive: leftPanelInteractive
                                flickableItem.flickableDirection: Flickable.VerticalFlick

                                style: MyScrollViewStyle {}
                                ListView {
                                    id: lv
                                    model: table_model
                                    delegate: Row {
                                        id: values_row
                                        width: parent.width
                                        height: row_height

                                        property string text_x: model.axis_x
                                        property string text_y: model.axis_y
                                        property bool is_remove_visible: !(model.is_start === "true");

                                        Rectangle{
                                            id: axis_x
                                            width: calibration_table.cell_width
                                            height: row_height
                                            color: table_std_color
                                            border.width: 1
                                            border.color: colorDialogBorder
                                            TextInput{
                                                id: text_axis_x
                                                width: parent.width
                                                height: parent.height
                                                horizontalAlignment: TextEdit.AlignHCenter
                                                verticalAlignment: TextEdit.AlignVCenter
                                                color: std_body
                                                font.pixelSize: is_remove_visible ? 15 : 12
                                                font.bold: true
                                                text: text_x
                                                validator: RegExpValidator { regExp:/[0-9]+/ }
                                                readOnly: !is_remove_visible
                                                Keys.onEnterPressed: {
                                                    // Перевод фокуса на воторой элемент в строке таблицы
                                                    setFocusToYCell(text_axis_x, text_axis_y, event.key);
                                                }
                                                Keys.onReturnPressed: {
                                                    // Перевод фокуса на воторой элемент в строке таблицы
                                                   setFocusToYCell(text_axis_x, text_axis_y, event.key);
                                                }
                                                onFocusChanged: {
                                                    if (true === focus) {
                                                        // сразу выделим все чтобы было удобно редактировать
                                                        selectAll();
                                                        parent.color = table_active_color;
                                                    } else {
                                                        console.log("===== text_axis_x focus loosed");
                                                        // принимаем последенее значение TextEdit
                                                        parent.color = table_std_color;
                                                        acceptXValue(model.index, text);
                                                        deselect();
                                                    }
                                                }
                                            }
                                            MouseArea{
                                                id: axis_x_mouse_area
                                                anchors.fill: parent;
                                                visible: is_remove_visible

                                                property bool is_double_clicked: false

                                                onClicked: {
                                                    console.log("===== axis_x_mouse_area clicked");
                                                    // Нужно чтобы выделилась клетка таблицы
                                                    setupCurrentCell(text_axis_x, model.index);

                                                    parent.color = table_gather_value_color;
                                                    focus = true;
                                                    is_double_clicked = false;
                                                    forceActiveFocus();
                                                }
                                                onFocusChanged: {
                                                    console.log("===== axis_x_mouse_area focus changed to ", focus);
                                                    if (false === focus) {
                                                        if (text_axis_x === current_cell) {
                                                            clearCurrentCell();
                                                        }
                                                        // Вернем старое значение
                                                        text_axis_x.text = model.axis_x;
                                                        if (!is_double_clicked) {
                                                            parent.color = table_std_color;
                                                        }

                                                    } else {
                                                        if (is_remove_visible) {
                                                            resetAvgValue();
                                                        }
                                                    }
                                                }
                                                onDoubleClicked: {
                                                    console.log("===== axis_x_mouse_area DOUBLE clicked");
                                                    is_double_clicked = true;
                                                    text_axis_x.forceActiveFocus();
                                                }
                                            }

                                        }
                                        Rectangle {
                                            id: axis_y
                                            width: calibration_table.cell_width
                                            height: row_height
                                            color: table_y_std_color
                                            border.width: 1
                                            border.color: colorDialogBorder
                                            TextInput{
                                                id: text_axis_y
                                                width: parent.width
                                                height: parent.height
                                                horizontalAlignment: TextEdit.AlignHCenter
                                                verticalAlignment: TextEdit.AlignVCenter
                                                color: std_body
                                                font.pixelSize: is_remove_visible ? 15 : 12
                                                font.bold: true
                                                text: text_y
                                                validator: RegExpValidator { regExp:/[0-9]+/ }
                                                readOnly: !is_remove_visible
                                                Keys.onEnterPressed: {
                                                    handleYreturnPressed(event.key, text_axis_y);
                                                }
                                                Keys.onReturnPressed: {
                                                    handleYreturnPressed(event.key, text_axis_y);
                                                }
                                                onFocusChanged: {
                                                    if (true === focus) {
                                                        // Здесь идея такая, что нужно установить текущее значение активной строки в ListView
                                                        lv.currentIndex = model.index;
                                                        // при этом у нас фокус может слететь(из-за игры с currentIndex), поэтому просто еще раз присваиваем фокус этому элементу.
                                                        focus = true;
                                                        // И сразу выделим текст чтобы его было удобно редактировать
                                                        selectAll();
                                                        parent.color = table_active_color;
                                                        // Если у нас не заголовок, то мы можем получить значение с выбранной переменноой, сделаем видимой кнопку для этого.

                                                    } else {
                                                        // Принимаем значение которое осталось в textEdit
                                                        acceptYValue(model.index, text);
                                                        // скрываем кнопку применения значения из переменной
                                                        deselect();
                                                        parent.color = table_y_std_color;
                                                    }
                                                }
                                            }
                                        }
                                        Rectangle {
                                            id: removeRect
                                            width: Math.round(18 * kDPI)
                                            height: Math.round(18 * kDPI)
                                            color: "transparent"

                                            Image {
                                                id: removeImage

                                                visible: is_remove_visible
                                                source: removeRectArea.containsMouse?"images/icon_close.svg":"images/icon_gray_close.svg"
                                                sourceSize.width: Math.round(18 * kDPI)
                                                sourceSize.height: Math.round(18 * kDPI)
                                                smooth: false

                                                MouseArea {
                                                    id: removeRectArea
                                                    objectName: "removeRectArea"
                                                    anchors.fill: parent
                                                    cursorShape: Qt.PointingHandCursor
                                                    hoverEnabled: true
                                                    onClicked: {
                                                        console.debug("fuelTarFilterDialog. Calibration table. remove data")
                                                        index_row_to_delete = model.index;
                                                        removeRowDialog.arg1 = values_row.text_x;
                                                        removeRowDialog.arg2 = values_row.text_y;
                                                        forceActiveFocus();
                                                        removeRowDialog.open();
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }

                    Rectangle{
                        id: plot_container
                        width: 400
                        height: 360

                        color: std_body

                        anchors.top: calibration_table_container.top

                        anchors.left: calibration_table_container.right
                        anchors.leftMargin: margin_offset*4

                        Rectangle{
                            id: plot

                            width: 400
                            height: 300

                            Loader{
                                anchors.fill: parent
                                sourceComponent: chart_view
                            }

                            ChartView {
                                id: view
                                anchors.fill: parent
                                antialiasing: true

                                AreaSeries {
                                    id: areaSer
                                    name: qsTr("Interpolation of Table");
                                    color: "transparent"
                                    borderColor: "#FF0039A5"
                                    borderWidth: 1

                                    axisX: ValueAxis {
                                        id: valueAxisX
                                        min: 0
                                        max: 100
                                        tickCount: 6
                                        labelFormat: "%.0f"
                                    }
                                    axisY: ValueAxis {
                                        id: valueAxisY
                                        min: 0
                                        max: 100
                                        tickCount: 6
                                        labelFormat: "%.0f"
                                    }
                                    upperSeries: LineSeries {
                                        id: plot_line
                                    }
                                }
                            }
                        }
                    }

                    // Proxy текст чтобы определить размер кнопок пользуясь определением размера шрифта
                    Text {
                        id: save_btt_txt
                        text: qsTr("Save Table")
                        anchors.centerIn: parent;
                        visible: false
                    }
                    Text {
                        id: load_btt_txt
                        text: qsTr("Load Table")
                        anchors.centerIn: parent;
                        visible: false
                    }
                    Button {
                        anchors.bottom: body.bottom
                        anchors.bottomMargin: margin_offset
                        anchors.left: body.left
                        anchors.leftMargin: margin_offset*2

                        id: save_table_button
                        text:  qsTr("Save Table");
                        style: ButtonStyle {
                            background: Rectangle {
                                implicitWidth: save_btt_txt.width + 20
                                implicitHeight: row_height
                                radius: row_height/4
                                border.width: control.activeFocus ? 2 : 1
                                border.color: "#888"
                                gradient: Gradient {
                                    GradientStop { position: 0 ; color: control.pressed ? "#ccc" : "#eee" }
                                    GradientStop { position: 1 ; color: control.pressed ? "#aaa" : "#ccc" }
                                }
                            }
                        }
                        onClicked: {
                            forceActiveFocus();
                            focus = true;
                            var name = fileIOQML.getSaveFileName();
                            console.debug("SaveFileName", name);
                            var data = serializeTableDataToCsv(table_model);
                            saveCalTableToFile(name, data);
                        }
                    }

                    Button {
                        anchors.bottom: body.bottom
                        anchors.bottomMargin: margin_offset
                        anchors.left: save_table_button.right
                        anchors.leftMargin: margin_offset*2

                        id: load_table_button
                        text:  qsTr("Load Table");
                        style: ButtonStyle {
                            background: Rectangle {
                                implicitWidth: load_btt_txt.width + 20
                                implicitHeight: row_height
                                radius: row_height/4
                                border.width: control.activeFocus ? 2 : 1
                                border.color: "#888"
                                gradient: Gradient {
                                    GradientStop { position: 0 ; color: control.pressed ? "#ccc" : "#eee" }
                                    GradientStop { position: 1 ; color: control.pressed ? "#aaa" : "#ccc" }
                                }
                            }
                        }
                        onClicked: {
                            forceActiveFocus();
                            var name = fileIOQML.getOpenFileName();
                            console.debug("OpenFileName", name);
                            if ("" !== name) {
                                var data = loadCalTableFromFile(name);
                                console.debug("Loaded Data: ", data);
                                deserializeFromCsv(data);
                            }
                        }
                    }

                }
            }
        }
    }

    function dump() {
        console.debug("---- fuelTarFilterDialog dump start:", objectName);
        var i;
        console.debug("parametersModel:", parametersModel);
        if (parametersModel !== null) {
            for (i = 0; i < parametersModel.count; ++i)
                console.debug(parametersModel.get(i).paramName,
                              parametersModel.get(i).paramType,
                              parametersModel.get(i).varName);
        }
        console.debug("---- fuelTarFilterDialog dump finish:", objectName);

    }

    function tryToStartTimer(){
        var is_fuel_selected = (parametersModel.get(0).varName !== "");
        var is_temp_selected = (parametersModel.get(1).varName !== "");

        if (is_fuel_selected || is_temp_selected) {
            // говорим что хотим получать данные
            diagramItem.want();
            load_val_timer.start();
        }
    }

    function showItem(item) {
        diagramItem = item
        if (diagramItem.destroying)
            diagramItem.destroying.connect(function(){
                fuelTarFilterDialog.destroy();
            });
        fuelTarFilterDialog.parametersModel = item.parametersModel;

        // Теперь проверим данные в новой модели. И запишем все что используем внутри.
        var count = parametersModel.count;

        // Заполним таблицу
        var first_element = Code.clone(table_model.get(0));
        table_model.clear();
        table_model.append(first_element);
        // Дальше если есть данные о таблице - записываем их
        for (var i = tar_table_model_parameter_start_pos; i < count-1; ++i) {
            var element_x = parametersModel.get(i);
            var element_y = parametersModel.get(++i);
            var new_element = {"axis_x":element_x.varName, "axis_y":element_y.varName}
            table_model.append(new_element);
        }
        // Нужно восстановить записанное значение температруы
        if (parametersModel.get(3).varName !== "") {
            accepted_temperature = parseInt(parametersModel.get(3).varName);
        }

        dump();

        // Теперь попробуем определить индекс вычисляемого параметра и установить команду для него
        setupParameterID(parametersModel.get(0).varName)
        tryToStartTimer();
        diagramItem.newParameterSetted.connect(tryToStartTimer);

        // center dialog to item
        var itemCenter = diagramContainer.mapFromItem(item, item.width/2, item.height/2);
        centerPointX = itemCenter.x;
        centerPointY = itemCenter.y;
        // set size to item and show,...
        scaleFactorX = diagramContainer.diagramField.itemWidth / width;
        scaleFactorY = diagramContainer.diagramField.itemHeight / height;
        // ...than animate to normal size
        centerPointX = parent.width/2
        centerPointY = parent.height/2
        scaleFactorX = 1.0;
        scaleFactorY = 1.0;
    }

    function clearParametersModel(){
        var count = parametersModel.count;
        var std_count = tar_table_model_parameter_start_pos + 1;
        if (count > std_count) {
            var amount = count - std_count;
            parametersModel.remove(tar_table_model_parameter_start_pos, amount);
        }
    }


    function accept() {
        console.debug(objectName, arguments.callee.name);

        // Установим значение температуры
        acceptLastModified();

        // Поскольку работали с данными parametersModel отдельно в table_model, то нужно входившие в нее данные удалить
        clearParametersModel();

        // Заполняем модель значениями таблицы.
        var count = table_model.count;
        var local_tar_table = [];
        var i;
        for (i = 1; i< count; ++i) {

            var axis_x_name = table_model.get(i).axis_x;
            var axis_y_name = table_model.get(i).axis_y;

            var axis_x_var = {"paramName": axis_x_name, "varName":axis_x_name, "paramType":"int", "varCategory": "const", "varBaseCategory": categoryConst};
            var axis_y_var = {"paramName": axis_y_name, "varName":axis_y_name, "paramType":"int", "varCategory": "const", "varBaseCategory": categoryConst};

            // Добавляем в модель данных диалога данные таблицы тарировки
            parametersModel.insert(parametersModel.count - 1, axis_x_var);
            parametersModel.insert(parametersModel.count - 1, axis_y_var);
            local_tar_table.push(parseInt(axis_x_name));
            local_tar_table.push(parseInt(axis_y_name));

            // Так же добавим эти переменные в модель переменных, чтобы они потом могли быть восстановлены
            // И чтобы мы потом не придумывали отдельный тип для них.
            if (varsModel.getVarId(axis_x_name, categoryConst) === -1) {
                varsModel.addVar(categoryConst, "int","Algorithms", axis_x_name /*name*/,""/*description*/, axis_x_name /*value*/);
            }
            if (varsModel.getVarId(axis_y_name, categoryConst) === -1) {
                varsModel.addVar(categoryConst, "int","Algorithms", axis_y_name /*name*/,""/*description*/, axis_y_name /*value*/);
            }
        }
        console.log("accepted_temperature:", accepted_temperature);

        var acc_temp_str = "%1".arg(accepted_temperature);
        parametersModel.set(3, {"varName": acc_temp_str, "varBaseCategory": categoryConst}); // recieved_temperature
        if (varsModel.getVarId(acc_temp_str, categoryConst) === -1) {
            varsModel.addVar(categoryConst, "int","Algorithms", acc_temp_str,"", acc_temp_str);
        }

        diagramItem.tar_table = Code.clone(local_tar_table);
    }

    // Эта функция вызывается когда выделяется объект на поле диаграммы
    function selectionChanged(item, mouseEvent, itemWasSelected) {
        console.debug(objectName, "selectionChanged()", item)
        if ((visible === false) || (item === fuelTarFilterDialog))
            return;

        accept();
        dump();
    }

    function addNewCalibratinData() {
        var param_count = table_model.count;
        console.debug(objectName, "addNewCalibratinData() count:", param_count);
        table_model.insert(param_count, {"axis_x":"0", "axis_y":"0"})
    }

    function removeCalibrationData(index) {
        console.debug(objectName, "removeCalibrationData() index:", index);
        table_model.remove(index);
    }

    function setFocusToYCell(item1, item2, key_event){
        console.log(key_event)
        item1.focus = false
        item2.focus = true;
        item2.selectAll();
    }

    function saveCalTableToFile(file_name, data){
        fileIOQML.write(file_name, data);
    }

    function loadCalTableFromFile(file_name) {
        var data = fileIOQML.load(file_name);
        return data;
    }

    function serializeTableDataToCsv(table_model){
        var data = "";
        var i;
        for (i = 1; i < table_model.count; ++i) {
          var x_str = table_model.get(i).axis_x;
          var y_str = table_model.get(i).axis_y;
          x_str.replace("\n","");
          y_str.replace("\n","");
          x_str.replace("\r","");
          y_str.replace("\r","");

          var end_line = "\n";
          data += x_str + ";" + y_str + end_line;
        }
        console.debug("serialized data:", data);
        return data;
    }

    function deserializeFromCsv(data) {
        data = data.replace(/\r/g,"");
        data = data.replace(/\n+/g,"\n");
        console.debug( data);
        var rows = data.split("\n");
        console.debug("rows :", rows);

        if (table_model.count !== 1){
            table_model.remove(1, table_model.count - 1);
        }

        var counter = 0;
        var i;
        for (i = 0; i < rows.length; ++i) {
            var cols = rows[i].split(";");
            var x_str = cols[0];
            var y_str = cols[1];            
            if (x_str !=="" && y_str !=="" ) {
                console.debug("cols :", cols, y_str.indexOf("\n"));
                x_str.replace(/\n/gm,"_");
                y_str.replace(/\n/gm,"_");
                table_model.append({"axis_x": x_str, "axis_y": y_str});
            }
        }
    }

    function acceptXValue(index, value){
        console.debug("acceptXValue:", value);
        table_model.set(index, {"axis_x": value})
    }
    function acceptYValue(index, value){
        console.debug("acceptYValue:", value);
        table_model.set(index, {"axis_y": value})
    }

    function changeLineSeries(data_model){
        if (data_model.count === 1) {
            return;
        }

        plot_line.clear();

        var x_data =[];
        var y_data =[];

        var i;
        for (i = 1; i < data_model.count; ++i) {
            var tx = data_model.get(i).axis_x;
            var ty = data_model.get(i).axis_y;
            // Нужно загружать только полные поля
            console.debug("model data x =", tx,"y =", ty);
            if (tx !=="" && ty !=="" ) {
                x_data.push(parseFloat(tx));
                y_data.push(parseFloat(ty));
            }
        }

        var sorted_x_data = [];
        for (i = 0; i < x_data.length; ++i) {
            sorted_x_data.push(x_data[i]);
        }

        var sorted_y_data = [];

        sorted_x_data.sort(compareNumericMinFirst);
        for (i = 0; i < sorted_x_data.length; ++i) {
            var comp_val = sorted_x_data[i];
            for (var j = 0; j < y_data.length; ++j) {
                if (comp_val === x_data[j]) {
                    sorted_y_data[i] = y_data[j] ;
                }
            }
        }

        for (i = 0; i < sorted_x_data.length; ++i) {
            var x = sorted_x_data[i];
            var y = sorted_y_data[i];
            plot_line.append(x, y);
        }
        if (sorted_x_data.length < 2) {
            return;
        }

        view.axisX(areaSer).min = sorted_x_data[0];
        view.axisX(areaSer).max = sorted_x_data[sorted_x_data.length-1];

        sorted_y_data.sort(compareNumericMinFirst);
        view.axisY(areaSer).min = sorted_y_data[0];
        view.axisY(areaSer).max = sorted_y_data[sorted_y_data.length-1];

        areaSer.lowerSeries = areaSer.upperSeries;

        console.debug(x_data);
        console.debug(y_data);
        console.debug(sorted_x_data);
        console.debug(sorted_y_data);

        console.debug(view.axisX(areaSer).min)
        console.debug(view.axisX(areaSer).max)
        console.debug(view.axisY(areaSer).min)
        console.debug(view.axisY(areaSer).max)
    }

    // Функции определения граничных значений для графика
    function setupBorders(plot_data){
        var model_data = plot_data;
        // Мы рисуем график.

        //0. Подготовим данные для удобного использования
        var x_data =[];
        var y_data =[];
        var i;
        for (i = 0; i < model_data.count; ++i) {
            x_data.push(model_data.get(i).axis_x);
            y_data.push(model_data.get(i).axis_y);
        }

//        1. Определимся с размером оси Х
        var max_x = getMax(x_data);
        var min_x = getMin(x_data);
//        2. Определимся с размером оси У
        var max_y = getMax(y_data);
        var min_y = getMin(y_data);

//       3. Запишем новые границы
        valueAxisX.min = min_x;
        valueAxisX.max = max_x;

        valueAxisY.min = min_y;
        valueAxisY.max = max_y;
    }

    function getMax(data_array){
        var tmp_data = data_array;
        tmp_data.sort(compareNumericMaxFirst);
        console.debug(tmp_data);
        return tmp_data[0];
    }

    function compareNumericMaxFirst(a, b) {
      if (a > b) return -1;
      if (a < b) return 1;
    }

    function getMin(data_array){
        var tmp_data = data_array;
        tmp_data.sort(compareNumericMinFirst);
        console.debug(tmp_data);
        return tmp_data[0];
    }

    function compareNumericMinFirst(a, b) {
      if (a > b) return 1;
      if (a < b) return -1;
    }

    function setupParameterID(param_name){
        var k;
        console.log("===== new param_name: ", param_name);
        for (k = 0; k < localConstVarsProxyModel.count; ++k) {
          if (localConstVarsProxyModel.get(k, "varName") === param_name) {
            loaded_value_index = k;
            console.log("===== new value index: ", loaded_value_index);
            return true;
          }
        }
        return false;
    }

    function setupRecievedValue(){
        if (diagramItem.is_received_value_new) {
            loaded_value = diagramItem.getReceivedValue();
            Avg.addAvgData(loaded_value);
            average_value = Avg.evalAvg();
            measurment_count = Avg.getMeasurementCount();
        }
        if (diagramItem.is_received_value_new_2) {
            recieved_temperature = diagramItem.getReceivedValue2();
        }
    }

    function handleYreturnPressed(event_key, text_axis_y) {
        console.log(event_key);
        text_axis_y.focus = false;
        // Если активен последний элемент представления, мы не переходим на следующий
        if (lv.currentIndex === lv.count-1){
            return;
        }
        // переход на следующий элемент пердставления.
        lv.incrementCurrentIndex();
        // Row         Rectangle   TextEdit
        lv.currentItem.children[0].children[0].focus = true;
    }

    function getTableLastXItem(){
        lv.currentIndex = lv.count - 1; // Устанавливаем в последний элемент
                  // Row         Rectangle   TextEdit
        return lv.currentItem.children[0].children[0];
    }

    function setupCurrentCellActive(){
           // Row         Rectangle   TextEdit
        lv.currentItem.children[0].children[1].focus = true;
        lv.currentItem.children[0].color = table_gather_value_color;
    }

    function setupCurrentCell(cell, index){
        current_cell = cell;
        current_cell_index = index;
    }

    function clearCurrentCell(){
        current_cell = null;
        current_cell_index = 0;
    }

    function resetAvgValue(){
        Avg.clearAvgData();
        pass_time = 0;
        average_value = 0;
        measurment_count = 0;
        loaded_value = 0;
    }


    function setupCellValue(cell){
        // Раз в секунду выделим весь текст.
        cell.selectAll();
        // Затем запишем в него текущее значение
        cell.text = "%1".arg(average_value);
        // Сделаем вычисления и если посчитаем что значение устоялось
        cell.deselect();
        // поменяем цвет квадрата
        if (Avg.isSustained()) {
            cell.parent.color = table_sustained_value_color;
        }
    }

    function acceptLastModified(){
        // Для того чтобы принять все модифицированные данные нужно просто по закрытии
        // изменить фокус, что вызовет принятие данных.
        // Просто последний выделенный элемент выберем, а потом снимем с него фокус, тогда все должно отработать.
        lv.currentItem.children[2].children[0].focus = true;
        lv.currentItem.children[2].children[0].focus = false;
    }

    function getNextParamPos(){
        var params_count = 2;
        if ((params_count - 1) === current_param_index_pos){
            current_param_index_pos = 0;
        } else {
            ++current_param_index_pos;
        }
    }


    Component {
        id: chart_view
        ChartView {
            anchors.fill: parent
            antialiasing: true

            AreaSeries {
                name: "Russian"
                color: "#FFD52B1E"
                borderColor: "#FF0039A5"
                borderWidth: 1
                axisX: ValueAxis {
                    id: valueAxisX
                    min: 0
                    max: 100
                    tickCount: 50
                    labelFormat: "%.0f"
                }
                axisY: ValueAxis {
                    id: valueAxisY
                    min: 0
                    max: 100
                    tickCount: 50
                    labelFormat: "%.0f"
                }
                upperSeries: LineSeries {
                    id: plot_line
                }
            }
        }
    }

    MessageDialog {
        id: removeRowDialog
        property string default_name: qsTr("Are you sure for delete this row?\n     %1 : %2").arg(removeRowDialog.arg1).arg(removeRowDialog.arg2);
        property string arg1: "";
        property string arg2: "";

        objectName: "clearScriptDialog";
        title: qsTr("Fuel Filter");
        text: default_name;
        buttons: StandardButton.Yes | StandardButton.No
        modality: Qt.ApplicationModal

        flags: dialogWindowFlags
        onYesClicked: {
            console.debug("yes");
            signals_handler.delete_table_row();
        }

        onNoClicked: {
            console.debug("no");
        }
    }

    transform: [
        Scale {
            origin.x: width/2
            origin.y: height/2
            xScale: scaleFactorX
            yScale: scaleFactorY
        },
        Translate {
            x: centerPointX - width/2
            y: centerPointY - height/2
        }
    ]
    Behavior on centerPointX {
        enabled: visible
        NumberAnimation {
            easing.type: Easing.Linear
        }
    }
    Behavior on centerPointY {
        enabled: visible
        NumberAnimation {
            easing.type: Easing.Linear
        }
    }

    Component.onCompleted: {
        diagramField.selectionChanged.connect(fuelTarFilterDialog.selectionChanged);
        diagramContainer.setupZ.connect(fuelTarFilterDialog.setupZ);
        console.debug(objectName, "comleted")
    }
    Component.onDestruction: {
        diagramField.selectionChanged.disconnect(fuelTarFilterDialog.selectionChanged)
        diagramContainer.setupZ.disconnect(fuelTarFilterDialog.setupZ);
        console.debug(objectName, "destructed")
    }
}
