import QtQuick 2.2
import Qt.labs.platform 1.0
import "../auxiliary.js" as Code
import "lineItemCreation.js" as Generator
import "filters_functions.js" as Flf // Некоторые обобщеные функции для применения фильтров
import "view_types.js" as Vtypes // типы фильтров. Глобальные переменные.

Item {
    id: diagramItem
    objectName: "diagramItem"

    property string filter_type: "filter"

    property bool created: false    // true after item has been dropped to field
    property bool hovered: false    // true while dragging over it
    property bool selected: false   // true when item is selected
    property point startPoint       // item coordinates before drag of whole selection
    property string actionColor:     colorAction
    property string actionTextColor: colorTextAction
    property string actionBorderColor: "blue"

    property int actionId: 0    // unique for every action
    property string actionName: "default"        // for save/load to/form file. Must be equal to filename
    property string actionTitle: qsTr("default") // for displaying in diagram
    property var parametersNames: []        // to print to log
    property var parametersVarIds: []       // 0x00000000..0xFFFFFFFF
    property var parametersVarCategories: []    // Var categories (defined in main.qml):
                                                // "const", "global", "localConst", "local" or "list"
    property ListModel parametersModel: ListModel {}
    property int permanentParametersCount: 0    // the number of permanent parameters: 0 - everyone are permanent
    property string nonPermanentParameterName   // the paramName for the added parameter
    property string nonPermanentParamType       // the paramType for the added parameter
    property string nonPermanentVarCategory     // the varCategory for the added parameter

    property string actionToolTip: qsTr("default")
    property int actionOutputCount: 1   // 2 - condition, 1 - action, 0 - finish
    property string actionType: "action"  // action, systemEvent or userEvent
    property string dialogType: "parameters"    // Type of the dialog to edit action parameters
                                                // "parameters" - ParametersDialog.qml
                                                // "script"     - ScriptDialog.qml
    property string parameterDialogSource: 'ParametersDialog.qml'   // source of the individual file of the parameter dialog
    property string tarTableDialogSource: 'TarTableDialog.qml'   // source of the individual file of the parameter dialog
    property string fuelTarFilterDialogSource: 'FuelTarFilterDialog.qml'   // source of the individual file of the parameter dialog

    property bool is_dialog_created: false // Флаг указывающий на то был ли создан диалог для этогофильтра

    property var outputLine: null       // not null when connected
    property var outputLineFalse: null  // not null when connected (condition only)
    property string outputLineFalsePosition // False output position: "right" or "left"

    property var inputLinesArray: []    // is not empty when line(s) are connected
    property color chainColor           // color of the chain actions borders
    property int pageIndex: 0           // page of the diagram which contains this

    property bool hasError: false    // true - if checkDiagram() have found the error

    property int itemRectMargin: Math.round(10 * kDPI)
    property real scaleFactor: 1.0

    property var buttonsItem: null  // action buttons menu

    property var view_types: Vtypes.view_types
    property int view_type: Vtypes.view_types.get_param // По этой переменной определяется компонент пердставления который будет загружен для данного объекта
    property int filter_width: Math.round(320*kDPI)
    property int filter_height: Math.round(96*kDPI)
    property int parameter_width: Math.round(152*kDPI)
    property int parameter_height: Math.round(64*kDPI)
    property string filter_border_color: "blue"
    property string parameter_border_color: "gold"
    property string end_filter_parameter_border_color: "green"

    // Добавим новое свойство, оно будет использоваться для передачи результирующего параметра
    // соседу по графу
    property var next_item: null
    property var prev_item: null
    // Это сделано, чтобы была возможность отключить Drag Point
    property var drag_point_source: drag_point_component

    property var listOfConnectedItems: [[null,first_connect_rect], [null,second_connect_rect]];
    // Показывает индекс какого пдключенного блока изменялся последним
    // Всегда инициализировать 0-м. Необходимо для работы передачи параметров в блоки с view_type отличным от 0.
    property int last_connected_item: 0;

    property int connect_pos_x: x + itemRectMargin;
    property int connect_pos_y: y + height/2;

    // Добавим свойства для хранения таблицы тарировки
    property var tar_table: [];
    // Параметр для определенеия сколько в модели данных изначально параметров
    // Остальные считаем, что будудт входить в данные tar_table.
    property int origin_param_count: 0
    property bool need_check_tar_table: true
    // Свойство для задания подписей к "входам"
    property var inputs_lables: ["val1", "val2"];

    signal itemIsAboutToDestroy(Item item)
    signal itemSelected(Item item, var mouseEvent)
    signal itemMoved(Item item, point point)
    signal itemReleased(Item item)
    signal filterGenerateNeewResult()
    signal destroying()

    onFilterGenerateNeewResult: {
        console.log("filterGenerateNeewResult signal recieved. Generate new name:");
        if (view_type === Vtypes.view_types.save_param) { // не генерируем переменную для сохранения параметра
            return
        }
        generateResultVariableName();
    }

    function generateResultVariableName(){
        // Мы намернено оставляем постоянную генерацию имен переменных потому что при копировании блока -
        // новые связи будут установлены вручную и нужно генерить новыые имена переменных, иначе будет неправильно работать
        // Восстановление алгоритма
        var res_var_name = Generator.generateVarName(diagramAction);
        console.log("new result name: ", res_var_name, parametersModel.count);
        parametersModel.set(parametersModel.count-1, {"varName":res_var_name, "varBaseCategory":categoryGlobal});
        if (varsModel.getVarId("1", categoryConst) === -1) {
            varsModel.addVar(categoryConst, "int","Algorithms", "1","", "1");
        }
    }


    width: 200 + itemRectMargin*2
    height: 200 + itemRectMargin*2
    z: (pageIndex === diagramContainer.pageIndex) ? zDiagramObject : zDiagramObjectOnHiddenPage
    opacity: created ? 1 : 0.5  // transparent while dragging from toolbar (creating)
    enabled: pageIndex === diagramContainer.pageIndex

    Drag.active: mouseArea.drag.active
    Drag.hotSpot.x: width / 2
    Drag.hotSpot.y: height / 2
    Drag.onDragStarted: { console.debug(objectName, "Drag.onDragStarted:") }

    transform: Scale { origin.x:0; origin.y:0; xScale:scaleFactor; yScale: scaleFactor}

    Component.onCompleted: {
        chainColor = colorChainDefault;
        diagramItem.itemSelected.connect(diagramContainer.diagramField.itemSelected)
        diagramContainer.diagramField.selectionChanged.connect(diagramItem.selectionChanged)
        diagramContainer.diagramField.selectionMoved.connect(diagramItem.selectionMoved)
        diagramContainer.diagramField.selectionReleased.connect(diagramItem.selectionReleased)
        diagramContainer.diagramField.selectionFinished.connect(diagramItem.selectionFinished)

        diagramItem.itemMoved.connect(diagramContainer.diagramField.itemMoved)
        diagramItem.itemReleased.connect(diagramContainer.diagramField.itemReleased)

        diagramContainer.diagramField.dropPosChanged.connect(diagramItem.dropPosChanged);

        diagramContainer.diagramField.dropFinished.connect(diagramItem.dropFinished);

        diagramItem.itemIsAboutToDestroy.connect(diagramContainer.diagramField.itemIsAboutToDestroy)
        console.debug(objectName, "completed")
    }
    Component.onDestruction: {
        destroyButtons();
        destroying();
        diagramItem.itemSelected.disconnect(diagramContainer.diagramField.itemSelected)
        diagramContainer.diagramField.selectionChanged.disconnect(diagramItem.selectionChanged)
        diagramContainer.diagramField.selectionMoved.disconnect(diagramItem.selectionMoved)
        diagramContainer.diagramField.selectionReleased.disconnect(diagramItem.selectionReleased)
        diagramContainer.diagramField.selectionFinished.disconnect(diagramItem.selectionFinished)

        diagramItem.itemMoved.disconnect(diagramContainer.diagramField.itemMoved)
        diagramItem.itemReleased.disconnect(diagramContainer.diagramField.itemReleased)

        diagramContainer.diagramField.dropPosChanged.disconnect(diagramItem.dropPosChanged);

        diagramContainer.diagramField.dropFinished.disconnect(diagramItem.dropFinished);

        diagramItem.itemIsAboutToDestroy.disconnect(diagramContainer.diagramField.itemIsAboutToDestroy)
        console.debug(objectName, "destructed")
    }

    onHeightChanged: {
        console.debug(objectName, "onHeightChanged:");
        itemReleased(diagramItem);  // to recalculate "down" line
    }

    function dump() {
        console.debug("---- action dump start",
                    objectName, actionName, pageIndex,
                    parametersVarIds, parametersVarCategories,
                    outputLine, outputLineFalse, inputLinesArray)
        var i;
        for (i = 0; i < parametersModel.count; ++i) {
            var param = parametersModel.get(i);
            console.debug("  param:", i, param.paramName, param.varName);
        }
        for (i in children)
            console.debug("  child:", i, children[i])
        console.debug("---- action dump finish")
    }

    function destroyItem() {
        itemSelected(diagramItem, undefined)
        itemIsAboutToDestroy(diagramItem);
        destroy();
        diagramContainer.diagramField.dumpChildren();
    }

    function dropPosChanged(item, dropPos) {
        if (actionType !== "action")
            return
        if (item === diagramItem)
            return;

        console.debug("dropPosChanged()", objectName, item.objectName, dropPos)

        var pointInThis = mapFromItem(diagramContainer.diagramField, dropPos.x, dropPos.y)
        var pointInDiagramItemRect = mapToItem(diagramItemRect, pointInThis.x, pointInThis.y)
        var diagramItemRectHovered = diagramItemRect.contains(Qt.point(pointInDiagramItemRect.x, pointInDiagramItemRect.y))
        diagramItem.hovered = diagramItemRectHovered
    }

    function dropFinished() {
        if (diagramItem.hovered) {

        }
        diagramItem.hovered = false
    }

    function selectionChanged(item, mouseEvent, itemWasSelected) {
        console.debug(objectName, "selectionChanged()", item)

        if (pageIndex !== diagramContainer.pageIndex)
            return;

        //! select self
        if (item === diagramItem) {
            if ((mouseEvent !== undefined)
                    && (mouseEvent.modifiers & Qt.ControlModifier))
            {
                diagramItem.selected = !diagramItem.selected
            }
            else
            {
                diagramItem.selected = true;
            }
            startPoint = Qt.point(diagramItem.x, diagramItem.y);

            if (diagramItem.selected === true)
                showButtonsDelayed();
            else
                hideButtonsDelayed();

            return;
        }
        else {
            hideButtonsDelayed();
        }

        //! select already selected item
        if (itemWasSelected)
            return;

        //! select new item (previously not selected)
        // don't release selection in case of Ctrl is pressed
        if ((mouseEvent !== undefined)
                && (mouseEvent.modifiers & Qt.ControlModifier))
            return;

        // deselect self at last
        diagramItem.selected = false;
    }

    function selectionMoved(item, deltaX, deltaY) {
        console.debug(objectName, "selectionMoved()", item, deltaX, deltaY)

        if (pageIndex !== diagramContainer.pageIndex)
            return;

        // ignore self
        if (item === diagramItem)
            return;
        if (diagramItem.selected === false)
            return;

        x = startPoint.x + deltaX;
        y = startPoint.y + deltaY;
        diagramContainer.diagramField.alignItem(diagramItem);
    }

    function selectionReleased(item) {
        console.debug(objectName, "selectionReleased()", item)

        if (pageIndex !== diagramContainer.pageIndex)
            return;

        if (diagramItem.selected)
            startPoint = Qt.point(diagramItem.x, diagramItem.y);
    }

    //! Select self if it is inside selectedRect
    function selectionFinished(selectedRect) {
        console.debug(objectName, "selectionFinished()", selectedRect);

        if (pageIndex !== diagramContainer.pageIndex)
            return;

        if ((selectedRect.x <= x)
                && (selectedRect.y <= y)
                && ((x + width) <= (selectedRect.x + selectedRect.width))
                && ((y + height) <= (selectedRect.y + selectedRect.height)))
        {
            selectionChanged(diagramItem, undefined, true);
        }
    }

    function showParametersDialog() {
        hasError = false;
        if (dialogType === "parameters")
            diagramContainer.loadAndCreateParametersWindow(diagramItem);
        else if(dialogType === "tar_table_filter")
            diagramContainer.showTarTableDialog(diagramItem);
        else if(dialogType === "fuel_tar_table_filter")
            diagramContainer.showFuelTarFilterDialog(diagramItem);
    }

    function showColorDialog() {
        colorDialog.open();
    }

    //! Update properties which is needed for saving
    function prepareForSaving() {
        console.debug(actionName, arguments.callee.name)
        parametersNames.length = 0;
        parametersVarIds.length = 0;
        parametersVarCategories.length = 0;
        // process every parameter
        var j, k
        for (j = 0; j < parametersModel.count; ++j) {
            var param = parametersModel.get(j)

            console.debug(" ", actionName, j, param.paramName, param.varName)

            parametersNames[j] = param.paramName

            // search index for list param
            if (param.paramType === categoryList) {
                for (k = 0; k < param.paramList.count; ++k)
                    if (param.paramList.get(k).varName === param.varName) {
                        parametersVarIds[j] = k;
                        parametersVarCategories[j] = categoryList;
                        break;
                    }
                if (k !== param.paramList.count) // varName was found
                    continue;
            }

            // Рассмотрим категорию canIndex
            if (param.paramType === categoryCanIndex) {
                parametersVarIds[j] = param.varName;
                parametersVarCategories[j] = categoryCanIndex;
                continue;
            }

            // search in const vars
            var id = diagramContainer.diagramField.getVarIdFromProxyModel(constVarsProxyModel, param)
            if (id !== -1)
            {
                parametersVarIds[j] = id;
                parametersVarCategories[j] = categoryConst;
                continue;
            }

            // search in global vars
            id = diagramContainer.diagramField.getVarIdFromProxyModel(globalVarsProxyModel, param)
            if (id !== -1)
            {
                parametersVarIds[j] = id;
                parametersVarCategories[j] = categoryGlobal;
                continue;
            }

            // search in localConst vars
            id = diagramContainer.diagramField.getVarIdFromProxyModel(localConstVarsProxyModel, param)
            if (id !== -1)
            {
                parametersVarIds[j] = id;
                parametersVarCategories[j] = categoryLocalConst;
                continue;
            }

            // search in local vars
            id = diagramContainer.diagramField.getVarIdFromProxyModel(localVarsProxyModel, param)
            if (id !== -1)
            {
                parametersVarIds[j] = id;
                parametersVarCategories[j] = categoryLocal;
                continue;
            }

            // varName not found
            parametersVarIds[j] = ""
            parametersVarCategories[j] = ""
        }
        console.debug(actionName, arguments.callee.name,
                      parametersNames, parametersVarIds,
                      parametersVarCategories)
    }

    // create buttons on enter
    function showButtonsDelayed() {
        if (buttonsItem === null)
            buttonsTimer.start();
        else
            buttonsItem.destroyTimer.stop();
    }

    // destroy buttons on exit
    function hideButtonsDelayed() {
        if (buttonsItem === null) {
            buttonsTimer.stop();
        }
        else {
            buttonsItem.destroyTimer.start();
        }
    }

    function createButtons () {
        if (!mouseArea.containsMouse)
            return;

        buttonsItem = Code.loadAndCreate("actions/ActionButtons.qml",
                                         diagramContainer.diagramField);
        buttonsItem.actionItem = diagramItem;
        buttonsItem.anchors.left = diagramItem.right;
        // Добавил отступ чтобы был доступ к точке соединения
        buttonsItem.anchors.leftMargin = 20;

        buttonsItem.anchors.verticalCenter = diagramItem.verticalCenter;
        buttonsItem.z = Qt.binding( function() {
            return (pageIndex === diagramContainer.pageIndex)
                    ? zVarPanel : zDiagramObjectOnHiddenPage
        })  // to hide when switch to another page
    }

    function destroyButtons() {
        buttonsTimer.stop();
        if (buttonsItem !== null)
            buttonsItem.destroyItem();
    }

    // **************************************************************************
    /// Блок функций для установки связей, определения и передачи параметров
    // **************************************************************************

    function setNext(obj) {
        // Установили потомка
        next_item = obj;
        // прицепили обработчик сигнала потомка к своему сигналу
        newValueRecieved.connect(next_item.newValueHandler);
        // установили себя как предка
        next_item.setPrevious(diagramItem);
        // установили
        sendResultToNext();
    }
    function setPrevious(obj){
        prev_item = obj;
    }

    // Функции организации получения данных от прибора
    function want(){
        switch(view_type) {
        case Vtypes.view_types.one_input_filter:
            if (null != prev_item) {
                prev_item.want();
            }
            break;
        case Vtypes.view_types.two_inputs_filter:
            if (null !== listOfConnectedItems[0][0]){
                listOfConnectedItems[0][0].want();
            }
            if (null !== listOfConnectedItems[1][0]){
                listOfConnectedItems[1][0].want();
            }
            break;
        case Vtypes.view_types.get_param:
            startValueLoader();
            break;
        }
    }
    function notWantAnymore(){
        switch(view_type) {
        case Vtypes.view_types.one_input_filter:
            if (null != prev_item) {
                prev_item.notWantAnymore();
            }
            break;
        case Vtypes.view_types.two_inputs_filter:
            if (null !== listOfConnectedItems[0][0]){
                listOfConnectedItems[0][0].notWantAnymore();
            }
            if (null !== listOfConnectedItems[1][0]){
                listOfConnectedItems[1][0].notWantAnymore();
            }
            break;
        case Vtypes.view_types.get_param:
            stopValueLoader();
            break;
        }
    }

    property int loaded_value_index: 0
    function setupParameterID(param_name){
        var k;
        console.log("===== new param_name: ", param_name);
        for (k = 0; k < localConstVarsProxyModel.count; ++k) {
          if (localConstVarsProxyModel.get(k, "varName") === param_name) {
            loaded_value_index = k + 0x4000;
            console.log("===== new value index: ", loaded_value_index);
            return true;
          }
        }
        return false;
    }

    function startValueLoader(){
        var model_elem = parametersModel.get(0);
        if (model_elem !== undefined) {
            setupParameterID(model_elem.varName);
        }
        variableLoader.setupCmd(loaded_value_index);
    }
    function stopValueLoader(){
        variableLoader.unsetCmd(loaded_value_index);
    }

    // Привязывается к сигналу от variableLoader, который отправляется когда получен набор значений от прибора
    Connections {
       target: variableLoader
       onValueReceived: {
           console.log("** receive value signal from loader:", diagramItem);
           // Подавляем этот сигнал для всех, кроме блоков получения параметров
           if (view_type !== Vtypes.view_types.get_param) {
               return;
           }
           setupRecievedValue();
       }
   }

    function setupRecievedValue(){
        var loaded_value = variableLoader.getValue(loaded_value_index);
        newValueRecieved(loaded_value, diagramItem);
    }

    signal newValueRecieved(int val, var item);
    // Храним полученные от прибора значения
    property int received_value: 0
    property int received_value_2: 0
    // Для  того чтобы диалог который будет это значение отображать знал изменилось оно или нет
    property bool is_received_value_new: false
    property bool is_received_value_new_2: false

    // Для синхронизации получения сигналов со значениями
    property int recv_counter: 0

    function setReceivedValue(value){
        is_received_value_new = true;
        ++recv_counter;
        received_value = value;
    }

    function getReceivedValue(){
        is_received_value_new = false;
        return received_value;
    }

    function setReceivedValue2(value){
        is_received_value_new_2 = true;
        ++recv_counter;
        received_value_2 = value;
    }

    function getReceivedValue2(){
        is_received_value_new_2 = false;
        return received_value_2;
    }

    function newValueHandler(value, item){
        switch(view_type){
            case Vtypes.view_types.one_input_filter:
                setReceivedValue(value);
                break;
            case Vtypes.view_types.two_inputs_filter: // 2 входа
                // Мы хотим ждать пока не получим значение на оба входа
                if (item === listOfConnectedItems[0][0]){
                    setReceivedValue(value);
                }
                if (item === listOfConnectedItems[1][0]){
                    setReceivedValue2(value);
                }
                // Установили 1 из 2-х значений.
                // Теперь посмотрим установлены ли уже оба. Если нет - дальше ничего не передаем
                if (recv_counter !== 2) {
                    return;
                }
                recv_counter = 0;
                break;
            default:
                break;
        }

        console.debug("onNewValueRecieved: ", diagramItem, "  value:", received_value, "  second value:", received_value_2);
        if (null !== next_item) {
            newValueRecieved(applyAction(value), diagramItem);
        }
    }

    function sendResultToNext(){
        if (null === next_item) return
        var result_pos = parametersModel.count-1;
        var result_element = Code.clone(parametersModel.get(result_pos)); // clone из axullary.js
        next_item.setNewParameter(result_element);
        next_item.newParameterSetted();// Отправляем сигнал что новый параметр установлен
    }
    signal newParameterSetted();
    function setNewParameter(res){
        var pos = last_connected_item; // Для всех типов view_type, кроме Vtypes.view_types.two_inputs_filter будет всегда 0.
        parametersModel.set(pos, res);
    }

    // Устанавливаем новый подключенный блок диаграммы
    function setConnectionObject(item,cx,cy){
        var dy = diagramItemRect.y + diagramItemRect.height/2;
        console.log("\\\\\\\\\\\\\\ getConnectionObject", cy, "  |  ", dy);

        // Добавим такую штуку что если оба условия выполняются, то подключение в 1-й или второй кубы.
        // А вот если тип не 4, то подключение всегда в первый.
        if ((cy > dy) && (view_type === Vtypes.view_types.two_inputs_filter)) {
            console.log("return second_connect_rect");
            last_connected_item = 1;
            listOfConnectedItems[1][0] = item;
            // Если он был подключен на друго порт, нужно его со старого отключить
            if (listOfConnectedItems[0][0] === item) {
                listOfConnectedItems[0][0] = null;
            }
        } else {
            console.log("return first_connect_rect");
            last_connected_item = 0;
            listOfConnectedItems[0][0] = item;
            // Если он был подключен на друго порт, нужно его со старого отключить
            if (listOfConnectedItems[1][0] === item) {
                listOfConnectedItems[1][0] = null;
            }
        }

        console.log(listOfConnectedItems, "last changed:", last_connected_item);
    }

    function getLastConnectedPort(){
        return listOfConnectedItems[last_connected_item][1];
    }

    function getPortByItem(item){
        if (listOfConnectedItems[0][0] === item) {
            return listOfConnectedItems[0][1];
        }
        if (listOfConnectedItems[1][0] === item) {
            return listOfConnectedItems[1][1];
        }
        return null;
    }

    function addInputLine(line){
        // Если пытаемся добавить линию. Делаем следующие шаги
        /// 1. Как только получили линию - запихиваем в inputLines.
        inputLinesArray.push(line);

        /// 2. Удалим Lines, которые не прицеплены к нашему блоку.
        var lines_to_destroy = [];
        inputLinesArray.forEach(function (ar_line, i, array){
            if (!isItemConnected(ar_line.startPoint)){
                lines_to_destroy.push(ar_line);
            }
        })

        lines_to_destroy.forEach(function (ar_line, i, array) {
            ar_line.destroyItem();
        });

        /// 3. Затем пробегаемеся по загруженным item. И выставляем им соответствующие Lines.
        var tmp_input_lines_array = [];
        var i = 0;
        for (i=0;i< listOfConnectedItems.length; ++i){
            var cur_item = listOfConnectedItems[i][0];
            for (var j = 0; j < inputLinesArray.length; ++j){
                var cur_line = inputLinesArray[j];
                if (undefined !== cur_line) {
                    var cur_Line_item = cur_line.startPoint;
                    if (cur_item === cur_Line_item) {
                        tmp_input_lines_array[i] = cur_line
                    }
                }
            }
        }

        inputLinesArray = tmp_input_lines_array;
    }

    function getPortPosByItem(item){
        if (listOfConnectedItems[0][0] === item) {
            return 0;
        }
        if (listOfConnectedItems[1][0] === item) {
            return 1;
        }
        return null;
    }

    function isItemConnected(item){
        console.log("*** ",listOfConnectedItems[0][0], listOfConnectedItems[1][0], " ____ ", item);
        if (listOfConnectedItems[0][0] === item || listOfConnectedItems[1][0] === item) {
            return true;
        }
        return false;
    }

    function restoreInputConnection(line, input_item){
        // Считаем что к моменту когда вызывается эта функция у нас уже
        // восстановлены параметры модели.
        var name_pos = input_item.parametersModel.count - 1;
        var item_var_name = input_item.parametersModel.get(name_pos).varName;
        for (var j = 0; j < parametersModel.count; ++j){
            console.log(j,":", parametersModel.get(j).varName);
        }

        for (var i=0; i<parametersModel.count - 1; ++i){
            var cur_var_name = parametersModel.get(i).varName;
            if (cur_var_name === item_var_name) {
                listOfConnectedItems[i][0] = input_item;
                inputLinesArray[i] = line;
                // Пометим последнюю подключенную линию
                last_connected_item = i;
                return; // выходим
            }
        }
    }

    // **************************************************************************
    /// Блок функций для определения имени подключенного датчика
    // **************************************************************************
    function getConnectedVarName(){
        switch(view_type) {
        case Vtypes.view_types.one_input_filter:
            if (null != prev_item) {
                return prev_item.getConnectedVarName();
            }
            break;
        case Vtypes.view_types.two_inputs_filter:
            var name1 = "???";
            var name2 = "???";
            if (null !== listOfConnectedItems[0][0]){
                name1 = listOfConnectedItems[0][0].getConnectedVarName();
            }
            if (null !== listOfConnectedItems[1][0]){
                name2 = listOfConnectedItems[1][0].getConnectedVarName();
            }
            return "%1, %2".arg(name1).arg(name2);
        case Vtypes.view_types.get_param:
            return parametersModel.get(0).varName;
        }
    }


    // **************************************************************************
    /// Блок функций для применения фильтров
    // **************************************************************************

    function fillTarTable(){
        var count = parametersModel.count;
        if (origin_param_count === 0 && count <= origin_param_count){
            return;
        }

        // Дальше если есть данные о таблице - записываем их
        for (var i = origin_param_count-1; i < count-1; ++i) {
            tar_table.push(parseInt(parametersModel.get(i).varName));
        }
    }

    function applyFuelFilter(){
        var temp = received_value_2; // температура принятая во втором параметре
        var temp0 = parseInt(parametersModel.get(3).varName); // это число записанное в значение темпертуры.
        var ro = Flf.findSelectedParameterValue(parametersModel.get(2));
        // Если таблица не установлена, а вызов произошел - нужно попробовать ее заполнить
        if (tar_table.length === 0) {
            fillTarTable();
        }
        var res_val = Flf.applyTarTableFilter(received_value, tar_table)*(1 + (temp - temp0)*Flf.koeffTE(ro));
        return res_val;
    }

    property int old_filtered_value: 0

    function applyMovingAverageFilter(value){
        var window_len = parseInt(parametersModel.get(1).varName);
        var res = Flf.average(value, old_filtered_value, window_len);
        return res;
    }

    function applyHighPassFilter(value){
        // Пока не реализован
        return value;
    }

    function applyMathFilter(left_op, right_op){
        var op_id = Flf.findSelectedParameterValue(parametersModel.get(2));
        var res = Flf.mathOperation(op_id, left_op, right_op);
        return res;
    }

    function applyAction(value){
        if (actionId ===  0x03){
            return Flf.applyTarTableFilter(value, tar_table);
        }
        if (actionId ===  0x04){
            return applyFuelFilter();
        }
        if (actionId === 0x01) {// moving average
            return applyMovingAverageFilter(value);
        }
        if (actionId === 0x02) {// high pass filter
            return applyHighPassFilter(value);
        }
        if (actionId === 0x06) {// math filter
            return applyMathFilter(received_value, received_value_2);
        }
        return value;
    }


    Timer {
        id: buttonsTimer

        interval: buttonsTimerInterval
        onTriggered: createButtons()
    }

    Rectangle {
        id: diagramItemVisualizer
        objectName: "diagramItemVisualizer"
        anchors.top: parent.top
        anchors.left: parent.left
        width: diagramItem.width
        height: diagramItem.height
        color: "orangered"
        opacity: 0.5
        visible: diagramItem.hasError
    }


    // Специальный невидимый прямоугольник для того,
    // чтобы к нему обращались методы создания соединений.
    // Они жестко завязаны на его id и objectName
    Rectangle {
        id: diagramItemRect
        objectName: "diagramItemRect"
        anchors.centerIn: parent
        width: parent.width - itemRectMargin*2
        height: parent.height - itemRectMargin*2
        color: "transparent"
        border.color: actionBorderColor
        border.width: regularThickness

        Loader {
            anchors.centerIn: parent
            sourceComponent: {
                switch(view_type) {
                case Vtypes.view_types.get_param:
                    diagramItem.width = diagramItem.parameter_width + itemRectMargin*2;
                    diagramItem.height = diagramItem.parameter_height + itemRectMargin*2;
                    diagramItem.actionBorderColor = diagramItem.parameter_border_color;
                    return parameter_view;
                case Vtypes.view_types.one_input_filter:
                    diagramItem.width = diagramItem.filter_width + itemRectMargin*2;
                    diagramItem.height = diagramItem.filter_height + itemRectMargin*2;
                    diagramItem.actionBorderColor = diagramItem.filter_border_color;
                    return filter_view;
                case Vtypes.view_types.save_param:
                    diagramItem.width = diagramItem.parameter_width + itemRectMargin*2;
                    diagramItem.height = diagramItem.parameter_height + itemRectMargin*2;
                    diagramItem.actionBorderColor = diagramItem.end_filter_parameter_border_color;
                    return parameter_view;
                case Vtypes.view_types.two_inputs_filter:
                    diagramItem.width = diagramItem.filter_width + itemRectMargin*2;
                    diagramItem.height = diagramItem.filter_height + itemRectMargin*2;
                    diagramItem.actionBorderColor = diagramItem.filter_border_color;
                    return filter_with_2_inputs_view;
                }
            }
        }
    }

    Rectangle {
        id: additionalSelectRectangle
        anchors.centerIn: parent
        width: diagramItemRect.width +4;
        height: diagramItemRect.height +4;
        color: "transparent"
        border.color: colorSelected
        border.width: regularThickness
        visible: false
    }


    Rectangle{
        id: first_connect_rect
        objectName: "first_connect_rect"
        visible: false
        property int connect_pos_x: parent.x + x;
        property int connect_pos_y: parent.y + y;
        width:4
        height:4
        color: "red"
        anchors.leftMargin: Math.round((-3)*kDPI);
        anchors.top: diagramItemRect.top
        anchors.topMargin: diagramItemRect.height/6;
        anchors.left: diagramItemRect.left;
    }

    Rectangle{
        id: second_connect_rect
        objectName: "second_connect_rect"
        visible: false
        property int connect_pos_x: parent.x + x;
        property int connect_pos_y: parent.y + y;
        width:4
        height:4

        color: "red"
        anchors.leftMargin: Math.round((-3)*kDPI);

        anchors.bottom: diagramItemRect.bottom
        anchors.bottomMargin: diagramItemRect.height/6 - height;
        anchors.left: diagramItemRect.left;
    }


    MouseArea {
        id: mouseArea
        objectName: "mouseArea"
        anchors.fill: parent

        drag.target: diagramItem
        drag.minimumX: -diagramItem.itemRectMargin
        drag.minimumY: -diagramItem.itemRectMargin
        drag.maximumX: diagramContainer.diagramField.width - diagramItem.width + diagramItem.itemRectMargin
        drag.maximumY: diagramContainer.diagramField.height - diagramItem.height + diagramItem.itemRectMargin

        hoverEnabled: true

        onEntered: showButtonsDelayed();
        onExited: hideButtonsDelayed();

        onPressed: {
            console.debug('mouseArea.onPressed:', diagramItem.x, diagramItem.y,
                          mouseX, mouseY)
            itemSelected(diagramItem, mouse)
        }
        onPositionChanged: {
            if (mouseArea.pressed) {
                itemMoved(diagramItem, Qt.point(mouseX, mouseY))
                diagramContainer.diagramField.alignItem(diagramItem)
            }
        }
        onReleased: {
            itemReleased(diagramItem);
            filterGenerateNeewResult();
        }
        onDoubleClicked: {
            if (parametersModel.count)
                showParametersDialog();
        }
        onHoveredChanged: {

        }

        // Это холдер ждя загрузчика точки соединения, чтобы внутри блока ее можно было удобно позиционировать
        Rectangle{
            id: drag_point_holder
            width: 8
            height: 8
            anchors.right: parent.right
            anchors.rightMargin: itemRectMargin*2
            anchors.verticalCenter: parent.verticalCenter
            color: "transparent"
            Loader {
                id: drag_point_loader
                sourceComponent: drag_point_source
            }
        }
    }


    ColorDialog {
        id: colorDialog
        objectName: "colorDialog"
        modality: Qt.ApplicationModal
        title: qsTr("Choose color for the chain")

        onAccepted: {
            console.log("Color:", colorDialog.color,
                        "SelectedColor:", colorDialog.currentColor);
            diagramContainer.diagramField.setChainColor(diagramItem, colorDialog.currentColor);
        }
        flags: dialogWindowFlags
    }

    Component{
        id: drag_point_component
        DragPoint {
            id: dragPointTrue
            objectName: "dragPointTrue"

            anchors.bottom: parent.bottom
            anchors.bottomMargin: itemRectMargin - radius
            anchors.horizontalCenter: parent.horizontalCenter

            dragItem: diagramItem
            visible: actionOutputCount
            textVisible: (actionType === "action") && (actionOutputCount == 2)   // only when it is Condition action
            pointType: pointTypeTrue
            outputLine: diagramItem.outputLine
            decoratorVisible: actionOutputCount && (mouseArea.containsMouse && !mouseArea.drag.active)
            showImage: actionOutputCount

            Component.onCompleted: {
                dragPosChanged.connect(diagramContainer.diagramField.dragPosChanged)
            }
            Component.onDestruction: {
                dragPosChanged.disconnect(diagramContainer.diagramField.dragPosChanged)
            }
        }
    }


    Component {
        id: parameter_view
        Rectangle {
            id: parameterItemRect
            objectName: "parameterItemRect"
            anchors.centerIn: parent
            color: actionColor

            Text {
                id: title
                text: diagramItem.actionTitle
                anchors.top: parent.top;
                anchors.left: parent.left;
                anchors.margins: 5
                onTextChanged: {
                    // обеспечивает увеличение размера блока если имя параметра не влезает
                    if ((title.width + 20) > (diagramItem.width - itemRectMargin*2)) {
                        diagramItem.width = title.width + 32 + itemRectMargin*2;
                    } else {
                        diagramItem.width = parameter_width + itemRectMargin*2;
                    }
                }
            }

            width: diagramItem.width - itemRectMargin*2 - 4;
            height: parameter_height - 4;
        }
    }

    Component {
        id: filter_view
        Rectangle {
            id: filterItemRect
            objectName: "filterItemRect"
            anchors.centerIn: parent
            color: actionColor

            Text {
                id: title
                text: diagramItem.actionTitle
                anchors.centerIn: parent
            }

            width: filter_width - 4
            height: filter_height - 4
        }
    }

    Component {
        id: filter_with_2_inputs_view
        Rectangle {
            id: filterItemRect
            objectName: "filterItemRect"
            anchors.centerIn: parent
            color: actionColor
            Text {
                id: title
                text: diagramItem.actionTitle
                anchors.centerIn: parent
                anchors.margins: 5
            }

            width: filter_width - 4
            height: filter_height - 4

            Rectangle {
                id: first_input
                color: "transparent"
                border.width: 1
                border.color: colorDiagramLine
                width: 20
                height: 2

                anchors.left: parent.left
                anchors.top: parent.top
                anchors.topMargin: parent.height/4 -10;
                Text{
                    height: 14
                    anchors.left: parent.right
                    anchors.leftMargin: 5
                    anchors.top: parent.top
                    anchors.topMargin: -height/2;
                    text: inputs_lables[0];
                }
            }
            Rectangle {
                id: second_input
                color: "transparent"
                border.width: 1
                border.color: colorDiagramLine
                width: 20
                height: 2

                anchors.left: parent.left
                anchors.bottom: parent.bottom
                anchors.bottomMargin: parent.height/4 -10;
                Text{
                    height: 14
                    anchors.left: parent.right
                    anchors.leftMargin: 5
                    anchors.top: parent.top
                    anchors.topMargin: -height/2;
                    text: inputs_lables[1];
                }
            }
        }
    }


    states: [
        State {
            name: "dragActive"
            extend: "selected"
            when: mouseArea.drag.active
            PropertyChanges { target: diagramItem; opacity: 0.5 }
            PropertyChanges { target: diagramItem; z: zDiagramDraggingObject }
        },
        State {
            name: "dropReady"
            when: hovered
            PropertyChanges { target: diagramItemRect; border.width: selectedThickness;  border.color: actionBorderColor}
        },
        State {
            name: "selected"
            when: selected
            PropertyChanges { target: diagramItemRect; border.color: colorSelected }
            PropertyChanges { target: diagramItemRect; border.width: selectedThickness;}
            PropertyChanges { target: additionalSelectRectangle; visible: true;}
        }
    ]
}
