import QtQuick 2.2
import QtQuick.Controls 1.4
import Qt.labs.platform 1.0
import QtQuick.Layouts 1.3
import com.galileosky.cppclasses 1.0
import "auxiliary.js" as Code

DropArea {
    id: diagramContainer
    objectName: "diagramContainer"

    property color deviceColorState: colorOther
    property color scriptColorState: colorOther
    property bool diagramFieldMoving: false
    property point pressedMousePoint: "0,0"
    property point startMovingPoint: "0,0"
    property int prevWidth: 0
    property int prevHeight: 0
    property real diagramCenterPointX: 0.0
    property real diagramCenterPointY: 0.0

    property real zTableDialog: zParameterDialog + 0.01;

    property string diagramType: "program"

    property bool diagramModified: false
    property DiagramField diagramField: DiagramField {
        parent: diagramContainer
        z: zDiagramField
        Component.onCompleted: {
            if (diagramType === "filter") {
                diagramField.is_filter = true;
            }
        }
    }
    property var parametersDialog: null
    property ScriptDialog scriptDialog: ScriptDialog {
        parent: diagramContainer
        z: zParameterDialog
    }

    property CanMessageParsingDialog canParserDialog: CanMessageParsingDialog{
        parent:  diagramContainer
        z: zParameterDialog
    }

    property var tarTableDialog: null;
    property var fuelTarFilterDialog: null;

    property alias pageIndex: pageView.pageIndex

    property string diagramNameUserData: qsTr("DiagName")
    property string diagramName: diagramNameUserData

    property var pageNames: []

    property real scaleFactorMin: 0.1
    property real scaleFactorMax: 2.0

    function centerDiagramField() {
        diagramCenterPointX = diagramContainer.width/2
        diagramCenterPointY = diagramContainer.height/2
        animateDiagramCenter.start()
    }

    function centerDiagramObjects() {
        // process every diagramObject_N on current page. obtaining bounds
        var i, object;
        var left = diagramField.width, top = diagramField.height, right = 0, bottom = 0;
        for (i = 0; i < diagramField.children.length; ++i) {
            object = diagramField.children[i];
            if (object.visible === false)
                continue;
            if (object.objectName.indexOf("diagramObject_") !== 0)
                continue;
            if (object.pageIndex !== pageIndex)
                continue;

            if (object.x < left)
                left = object.x;
            if (right < object.x + object.width)
                right = object.x + object.width;
            if (object.y < top)
                top = object.y;
            if (bottom < object.y + object.height)
                bottom = object.y + object.height;
        }

        var objectsCenterX = (right + left) / 2;
        var objectsCenterY = (top + bottom) / 2;
        var objectsCenter = diagramField.mapToItem(diagramContainer, objectsCenterX, objectsCenterY)
        var diagramCenter = diagramField.mapToItem(diagramContainer, diagramField.width/2, diagramField.height/2)

        diagramCenterPointX = diagramContainer.width/2 + (diagramCenter.x - objectsCenter.x)
        diagramCenterPointY = diagramContainer.height/2 + (diagramCenter.y - objectsCenter.y)
        animateDiagramCenter.start()
    }

    function moveDiagram(mouse) {
        var oldCenterPointX = diagramField.diagramCenterPointX
        var oldCenterPointY = diagramField.diagramCenterPointY

        // calculate new center
        var x = startMovingPoint.x + mouse.x - pressedMousePoint.x
        var y = startMovingPoint.y + mouse.y - pressedMousePoint.y
        var deltaX = x - oldCenterPointX
        var deltaY = y - oldCenterPointY

        // check new position. restore previous or moving
        var p0 = diagramField.mapToItem(diagramContainer, 0, 0)
        var p1 = diagramField.mapToItem(diagramContainer, diagramField.width, diagramField.height)
        if (((p1.x+deltaX) < width*0.1) || (width*0.9 < (p0.x+deltaX)))
            diagramField.diagramCenterPointX = oldCenterPointX
        else
            diagramField.diagramCenterPointX = x
        if (((p1.y+deltaY) < height*0.1) || (height*0.9 < (p0.y+deltaY)))
            diagramField.diagramCenterPointY = oldCenterPointY
        else
            diagramField.diagramCenterPointY = y

        console.debug(parent.objectName, "moveDiagram",
                      "leftTop:", Math.round(p0.x), Math.round(p0.y),
                      "rightBottom:", Math.round(p1.x), Math.round(p1.y),
                      "newCenter:", Math.round(x), Math.round(y),
                      "(", width, height, ")",
                      "resultCenter:",
                      Math.round(diagramField.diagramCenterPointX),
                      Math.round(diagramField.diagramCenterPointY))
    }

    function prepareForSaving() {
        // store pages model
        pageNames.length = 0;
        for (var i = 0; i < pageView.pageCount; ++i) {
            pageNames[i] = pageView.pageModel.get(i).pageName;
        }
        console.debug(arguments.callee.name, pageNames);
    }

    function restoreFromSaving() {
        // restore pages model
        pageView.pageModel.clear();
        pageNames.forEach(function (element, index, array) {
            pageView.pageModel.set(index, {"pageName" : element});
        });

        var filter_name = ".filter"
        var index_filter_extension = diagramName.indexOf(filter_name)

        if ( -1 === index_filter_extension ) {
			/* если имя скрипта НЕ содержит расширения '.filter', просто выводим название */
            diagramNameUserData = diagramName
        } else {
            /* из строки типа 'test_script.filter' получаем 'test_script' */
            diagramNameUserData = diagramName.substring(0, index_filter_extension)
        }
    }

    function loadAndCreateParametersWindow(item) {
        if (parametersDialog != null)
            parametersDialog.destroyDialog();

        if (item.parameterDialogSource !== "none.qml") {

            parametersDialog = Code.loadAndCreate(item.parameterDialogSource,
                                              diagramContainer);
            if (parametersDialog === null) {
                console.error(arguments.callee.name, "Object creation error");
                return;
            }

            parametersDialog.showItem(item)
        }
    }

    function showScriptDialog(item) {
        console.debug(arguments.callee.name);
        scriptDialog.showItem(item);
    }

    function showCanParserDialog(item) {
        console.debug(arguments.callee.name);
        canParserDialog.showItem(item);
    }

    signal setupZ(var item);
    function handleDialogPressed(item){
        setupZ(item);
    }

    function showTarTableDialog(item) {
        if (!item.is_dialog_created) {
            console.debug(arguments.callee.name);
            // Здесь нужно не просто взять и запустить. А создать новый экземпляр.
            tarTableDialog = Code.loadAndCreate(item.tarTableDialogSource,
                                                  diagramContainer);
            tarTableDialog.z = zParameterDialog + 1;
            item.is_dialog_created = true;
            tarTableDialog.thisDialogPressed.connect(diagramContainer.handleDialogPressed);
            tarTableDialog.showItem(item);
        }
    }

    function showFuelTarFilterDialog(item){
        if (!item.is_dialog_created) {
            console.debug(arguments.callee.name);
            // Здесь нужно не просто взять и запустить. А создать новый экземпляр.
            fuelTarFilterDialog = Code.loadAndCreate(item.fuelTarFilterDialogSource,
                                                  diagramContainer);
            fuelTarFilterDialog.z = zParameterDialog + 1;
            item.is_dialog_created = true;
            fuelTarFilterDialog.thisDialogPressed.connect(diagramContainer.handleDialogPressed);
            fuelTarFilterDialog.showItem(item);
        }
    }

    function showSaveDialog() {
        saveAlgorithm();
    }

    function saveAlgorithm() {
        if (false === checkAlgorithmName()) {
            return false;
        }

        prepareForSaving();
        diagramField.prepareForSaving()

        if (!diagramField.checkDiagram()) {
            diagramErrorOnSave.open()
            return false;
        }

        if(!saveLocalRemote(true))
            return false;

        scriptGenerator.scriptSavedFromQML();
        return true;
    }

    function loadAlgorithmUSB() {
        if (!checkAlgorithmName())
            return;
        if(scriptGenerator.checkUnsavedChanges())
        {
            if (!saveAlgorithm())
                return;
        }
        diagramField.dumpChildren();
        if (!diagramField.checkDiagram())
            return;
        uploadAlgorithm(true)
    }

    function loadBinaryAlgorithm()
    {
        if (!scriptGenerator.readScriptFromBinaryFile(diagramContainer))
            return;
        diagramNameUserData = diagramContainer.diagramName;
        if (checkAlgorithmName())
            uploadAlgorithm(false)
    }

    function uploadAlgorithm(isNeedGenerateAlgorithm)
    {
        scriptGenerator.setIsNeedGenerateAlgorithm(isNeedGenerateAlgorithm);
        if (diagramType == "filter") {
            continueUploadAlgorithm(false);
        } else {
            clearScriptDialog.open();
        }
    }

    function returnYes(){
        continueUploadAlgorithm(true);
    }
    function returnNo(){
        continueUploadAlgorithm(false);
    }

    function continueUploadAlgorithm(is_clear_needed){
        prepareForSaving();
        scriptGenerator.uploadAlgorithmUSB(diagramField, diagramContainer, diagramName, is_clear_needed);
        scriptGenerator.diagnosticsChecked = true;
        if (null !== statusPanel.diagnosticsButton)
            statusPanel.diagnosticsButton.checked = true;
    }

    function checkAlgorithmName(){

        if (diagramNameUserData === qsTr("DiagName")) {
            renameDiagramInfoDialog.text = renameDiagramInfoDialog.default_name
            renameDiagramInfoDialog.open();
            return false;
        }
        if (diagramNameUserData === "Nothing"
                || diagramNameUserData === "nothing"
                || diagramNameUserData === "SCRIPT")
        {
            renameDiagramInfoDialog.text = renameDiagramInfoDialog.unavailable_name
            renameDiagramInfoDialog.open();
            return false;
        }
        if (diagramNameUserData.length == 0) {
            emptyDiagramInfoDialog.open();
            return false;
        }
        var filter_name = ".filter";
        var search_pos = diagramNameUserData.length - filter_name.length;
        var index_filter_extension = diagramNameUserData.indexOf(filter_name, search_pos);

        if ( diagramType == "program" ) {
            /* если пользователь задал скрипту имя типа 'script.filter', считаем это за ошибку */
            if ( -1 !== index_filter_extension ) {
                renameDiagramInfoDialog.text = renameDiagramInfoDialog.filter_part_in_script
                renameDiagramInfoDialog.open();
                return false;
            }
            diagramName = diagramNameUserData
        }
        if ( diagramType == "filter" ) {
            diagramName = diagramNameUserData;
            /* если пользователь НЕ добавил в имя скрипта фильтра расширение, то добавим его */
            if ( -1 === index_filter_extension ) { diagramName += filter_name; }
        }
        return true;
    }

    function saveLocalRemote(isChecked) {
        if(scriptGenerator.saveLocalRemote(diagramField,
                                                  diagramContainer,
                                                  isChecked))
        {
            diagramModified = false;
            return true;
        }

        return false;
    }

    function openAlgorithm() {
        console.debug(objectName, "openAlgorithm()")
        if(scriptGenerator.checkUnsavedChanges())
        {
            if(!saveAlgorithm())
                return;
        }
        if (scriptGenerator.loadAlgorithm(diagramField, diagramContainer)) {
            centerDiagramObjects()
            diagramModified = false;
            scriptGenerator.scriptSavedFromQML();
            scriptGenerator.pushState(diagramContainer)
        }
    }

    function clearDiagram() {
        diagramField.selectionChanged(diagramField, undefined, false);
        diagramField.hideAllFieldObjects()
        diagramField.destroyHiddenFieldObjects()
        diagramNameUserData = qsTr("DiagName");
        diagramName = diagramNameUserData;
        pageView.pageModel.clear();
        pageView.pageModel.append({"pageName" : "<DEFAULT>"});
        scriptGenerator.removeAllVariables();
        diagramModified = false;
        scriptGenerator.pushState(diagramContainer);
    }

    function deleteAlgorithmDialog() {
        scriptGenerator.showDeleteAlgorithmDialog();
    }


    function actionMove(key) {
        var selectedObjects = diagramField.getSelectedArray();
        selectedObjects.forEach(function (element, index, array) {
            // move only objects. skip lines.
            if (element.objectName.indexOf("diagramObject_") === -1)
                return;

            if (key === Qt.Key_Left)
                element.x -= diagramField.objectsGridSize2x;
            else if (key === Qt.Key_Right)
                element.x += diagramField.objectsGridSize2x;
            else if (key === Qt.Key_Up)
                element.y -= diagramField.objectsGridSize2x;
            else if (key === Qt.Key_Down)
                element.y += diagramField.objectsGridSize2x;
        });
        diagramField.itemReleased(selectedObjects[0]);
    }

    function actionDelete() {
        var selectedObjects = diagramField.getSelectedArray();
        diagramField.destroyItems(selectedObjects);
    }

    function actionCopy() {
        diagramField.prepareForSaving();
        scriptGenerator.copyToBuffer(diagramField,
                                     diagramContainer.pageIndex);
    }

    function actionPaste() {
        diagramField.selectionChanged(diagramField, undefined, false);
        scriptGenerator.pasteFromBuffer(diagramField,
                                        diagramContainer.pageIndex);
        scriptGenerator.pushState(diagramContainer)
    }

    function actionUndo()
    {
        if (diagramType === "filter") {
            return;
        }
        scriptGenerator.undo(diagramContainer);
    }

    function actionRedo()
    {
        if (diagramType === "filter") {
            return;
        }
        scriptGenerator.redo(diagramContainer);
    }

    function findFocusedElement(element) {
        if (!element)
            element = diagramContainer;
        var i, f;
        // такая последовательность потому что activeFocus может быть у элемента и у его детей, надо найти крайнего
        for (i = 0; i < element.children.length; ++i) {
           f = findFocusedElement(element.children[i]);
           if (f)
               return f;
        }
        if (element.activeFocus)
            return element;
        return false;
    }

    function isDialogContent(element) {
        if (!element || element === diagramContainer)
            return false;
        if (element instanceof BaseParameterDialog)
            return true;
        return isDialogContent(element.parent);
    }


    function keysHandler(keyEvent) {
        console.debug(objectName, "keysHandler()",
                      keyEvent.text, keyEvent.key, keyEvent.modifiers);
        switch (keyEvent.key) {
            case Qt.Key_Left:
            case Qt.Key_Right:
            case Qt.Key_Up:
            case Qt.Key_Down:
                // можно проверить является ли элемент полем для текста if (findFocusedElement() instanceof TextInput) {
                // но лучше так
                if (!isDialogContent(findFocusedElement()))
                    actionMove(keyEvent.key);
                keyEvent.accepted = true;
                break;
            case Qt.Key_Escape:
                if (scriptGenerator.tutorialsShown) {
                    scriptGenerator.setTutorialsShown(false);
                    keyEvent.accepted = true;
                }
                break;
            case Qt.Key_Delete:
                actionDelete();
                keyEvent.accepted = true;
                break;
            case Qt.Key_C:
                if (keyEvent.modifiers & Qt.ControlModifier) {
                    actionCopy();
                    keyEvent.accepted = true;
                }
                break;
            case Qt.Key_V:
                if (keyEvent.modifiers & Qt.ControlModifier) {
                    actionPaste();
                    keyEvent.accepted = true;
                }
                break;
            case Qt.Key_Y:
                if (keyEvent.modifiers & Qt.ControlModifier)
                {
                    actionRedo();
                    keyEvent.accepted = true;
                }
                break;
            case Qt.Key_Z:
                if ((keyEvent.modifiers & Qt.ControlModifier) && (keyEvent.modifiers & Qt.ShiftModifier))
                {
                    actionRedo();
                    keyEvent.accepted = true;
                }
                else if (keyEvent.modifiers & Qt.ControlModifier)
                {
                    actionUndo();
                    keyEvent.accepted = true;
                }
                break;

            default:
                console.debug("Unhandled key");
        }
    }

    clip: true
    focus: true

    onWidthChanged: {
        diagramField.diagramCenterPointX =
                    diagramField.diagramCenterPointX + (width-prevWidth)/2.0
        prevWidth = width
    }
    onHeightChanged: {
        diagramField.diagramCenterPointY =
                diagramField.diagramCenterPointY + (height-prevHeight)/2.0
        prevHeight = height
    }

    Keys.onPressed: keysHandler(event)

    Menu {
        id: fieldMenu
        objectName: "fieldMenu"

        MenuItem {
            text: qsTr("Copy")
            onTriggered: {
                console.debug("Menu->Copy triggered");
                actionCopy();
            }
        }
        MenuItem {
            text: qsTr("Paste")
            onTriggered: {
                console.debug("Menu->Paste triggered");
                actionPaste();
            }
        }
    }

    MouseArea {
        id: mouseArea
        objectName: "mouseArea"

        anchors.fill: parent
        z: zDiagramMouseArea
        hoverEnabled: true
        acceptedButtons: Qt.LeftButton | Qt.MiddleButton | Qt.RightButton

        onPressed: {
            console.debug(parent.objectName, "onPressed", mouse.button)
            diagramContainer.forceActiveFocus();    /// grab focus from anyone
            if (mouse.button & Qt.RightButton) {
                diagramFieldMoving = true
                pressedMousePoint = Qt.point(mouse.x, mouse.y)
                startMovingPoint = Qt.point(diagramField.diagramCenterPointX,
                                            diagramField.diagramCenterPointY)
            }
        }
        onPositionChanged: {
            if (diagramFieldMoving)
                moveDiagram(mouse)
        }
        onReleased: {
            console.debug(parent.objectName, "onReleased", mouse.button)
            diagramFieldMoving = false;

            if (mouse.button & Qt.RightButton) {
                if ((Math.abs(pressedMousePoint.x - mouse.x) < 5 )
                        && (Math.abs(pressedMousePoint.y - mouse.y) < 5 ))
                {
                    fieldMenu.popup();
                }
            }
        }
        onClicked: {
            console.debug(parent.objectName, "onClicked", mouse.button)
            if (mouse.button & Qt.MiddleButton)
                centerDiagramObjects();
        }

        onWheel: {
            console.debug(parent.objectName, "onWheel:",
                          wheel.x, wheel.y, wheel.modifiers, wheel.angleDelta)
            // Scaling
            var controlPoint = Qt.point(wheel.x, wheel.y)
            var fieldPoint = mapToItem(diagramField, controlPoint.x, controlPoint.y)

            if (null !== statusPanel.scaleFactorSlider){
                if ((0 < wheel.angleDelta.y))
                    statusPanel.scaleFactorSlider.increase()
                else
                    statusPanel.scaleFactorSlider.decrease()
            }

            var resultPoint = mapFromItem(diagramField, fieldPoint.x, fieldPoint.y)
            diagramField.diagramCenterPointX -= resultPoint.x - controlPoint.x
            diagramField.diagramCenterPointY -= resultPoint.y - controlPoint.y
        }
    }

    Rectangle {
        id: toolbarRect
        objectName: "toolbarRect"

        width: parent.width
        height: toolbarHeight
        z: zToolbar
        color: colorPanel

        Rectangle {
            id: flowRect

            width: parent.width
            height: parent.height - pageView.height
            color: parent.color

            anchors.left: parent.left
            anchors.top: parent.top
            anchors.right: parent.right

            Flow {
                id: buttonLayout
                objectName: "buttonLayout"

                anchors.verticalCenter:  flowRect.verticalCenter
                anchors.left: parent.left
                anchors.right: parent.right
                spacing: panelCaptionHeight/10

                property real defaultWithButton: saveBinaryButton.implicitWidth * 4 + undoButton.width * 2 + saveBinaryButton.implicitWidth * 3
                property real buttonWithSum: defaultWithButton

                signal changeWithButton(real step)

                Component.onCompleted: {
                    changeWithButton.connect(saveButton.changeWidth)
                    changeWithButton.connect(openButton.changeWidth)
                    changeWithButton.connect(deleteButton.changeWidth)
                    changeWithButton.connect(clearButton.changeWidth)
                    changeWithButton.connect(uploadBinaryButton.changeWidth)
                    changeWithButton.connect(generateButton.changeWidth)
                    changeWithButton.connect(saveBinaryButton.changeWidth)
                }

                move: Transition { NumberAnimation { properties: "x"; easing.type: Easing.OutQuad } }

                onWidthChanged: {
                    buttonWithSum = saveButton.width + openButton.width + deleteButton.width + clearButton.width +
                            + undoButton.implicitWidth + redoButton.implicitWidth + saveBinaryButton.width +
                            + uploadBinaryButton.width + generateButton.width;

                    if (defaultWithButton / 2 >= buttonLayout.width)
                        changeWithButton(0);

                    if (buttonWithSum >= buttonLayout.width)
                        changeWithButton(-1);
                    else
                        changeWithButton(1);
                }


                ToolbarButton {
                    id: saveButton
                    objectName: "saveButton"

                    text: qsTr("Save")
                    tooltip: qsTr("Save algorithm source to file or upload source and binary to server")

                    onClicked: {
                        console.debug(objectName, "onClicked:")
                        showSaveDialog();
                    }
                }
                ToolbarButton {
                    id: openButton
                    objectName: "openButton"

                    text: qsTr("Open")
                    tooltip: qsTr("Open algorithm source from local file or server")

                    onClicked: {
                        console.debug(objectName, "onClicked:")
                        openAlgorithm();
                    }
                }
                ToolbarButton {
                    id: deleteButton
                    objectName: "deleteButton"

                    text: qsTr("Delete")
                    tooltip: qsTr("Select and delete some algorithm from server")

                    enabled: scriptGenerator.isEstablishedServer

                    onClicked: {
                        deleteAlgorithmDialog()
                    }
                }
                ToolbarButton {
                    id: clearButton
                    objectName: "clearButton"

                    text: qsTr("Clear")
                    tooltip: qsTr("Clear current workspace");

                    onClicked: {
                        console.debug(objectName, "onClicked:")
                        clearConfirmationDialog.open();
                    }
                }
                ImageButton {
                    id: undoButton
                    objectName: "undoButton"
                    imageSource:  "images/icon_undo.svg"
                    onClicked: {
                        actionUndo()
                    }
                }
                ImageButton {
                    id: redoButton
                    objectName: "redoButton"
                    imageSource:  "images/icon_redo.svg"
                    onClicked: {
                        actionRedo()
                    }
                }

                ToolbarButton {
                    id: saveBinaryButton
                    objectName: "saveBinaryButton"

                    text: qsTr("Save Binary")
                    tooltip: qsTr("Generate algorithm and save it locally")

                    implicitWidth: Math.round(160 * kDPI)
                    minWidth: Math.round(150 * kDPI)

                    onClicked: {
                        console.debug(objectName, "onClicked: saveBinary");
                        if(!checkAlgorithmName())
                            return;
                        diagramField.dumpChildren();
                        if (diagramField.checkDiagram()) {
                            prepareForSaving();
                            scriptGenerator.saveBinaryToFile(diagramField, diagramContainer, diagramName);
                        }
                        else {
                            console.log("Error in script, cancelled.")
                        }
                    }
                }

                ToolbarButton {
                    id: uploadBinaryButton
                    objectName: "uploadBinaryButton"

                    text: qsTr("Upload Binary")
                    tooltip: qsTr("Upload an algorithm to local USB device")

                    enabled: scriptGenerator.isUSBConnectedDevice

                    implicitWidth: Math.round(160 * kDPI)
                    minWidth: Math.round(150 * kDPI)

                    onClicked: {
                        console.debug(objectName, "onClicked: uploadBinary");
                        loadBinaryAlgorithm();
                    }
                }

                ToolbarButton {
                    id: generateButton
                    objectName: "generateButton"

                    text: qsTr("USB upload")
                    tooltip: qsTr("Generate algoritm and upload it to local USB device")

                    enabled: scriptGenerator.isUSBConnectedDevice

                    implicitWidth: Math.round(160 * kDPI)
                    minWidth: Math.round(130 * kDPI)

                    onClicked: {
                        console.debug(objectName, "onClicked:")
                        if (loadAlgorithmUSB())
                            uploadAlgorithm(true)
                    }
                }
            }
        }

        PageView {
            id: pageView
            objectName: "pageView"

            height: pageViewHeight - 5
            anchors.left: parent.left
            anchors.right: parent.right
            anchors.bottom: parent.bottom
        }
    }

    Rectangle {
        id: diagramContainerRect
        objectName: "diagramContainerRect"

        anchors.left: parent.left
        anchors.right: parent.right
        anchors.top: toolbarRect.bottom
        anchors.bottom: parent.bottom
        anchors.bottomMargin: -1
        z: zDiagramContainerRect
        color: colorDiagramBackground
    }
    Rectangle {
        id: diagramFieldBorderRect
        objectName: "diagramFieldBorderRect"

        anchors.fill: diagramContainerRect
        z: zToolbar
        color: "transparent"
        border.width: 1
        border.color: colorBorder
    }

    // For debug purpose only
//    DebugVarsPanel {
//        id: debugVarsPanel
//        objectName: "debugVarsPanel"

//        anchors.left: parent.left
//        anchors.right: parent.right
//        anchors.bottom: parent.botton
//    }

    MessageDialog {
        id: clearConfirmationDialog
        objectName: "clearConfirmationDialog"
        title: qsTr("GalileoSkyScripter")
        text: qsTr("Are you sure to clear everything?")
        informativeText: qsTr("Every item of every algorithm of every page will be deleted.")
        buttons: StandardButton.Yes | StandardButton.No
        modality: Qt.ApplicationModal
        onYesClicked: clearDiagram()
        flags: dialogWindowFlags
    }

    MessageDialog {
        id: renameDiagramInfoDialog
        property string default_name: qsTr("Default diagram name is detected.\n" +
                                           "Specify the unique diagram name instead of deault one.");
        property string filter_part_in_script: qsTr("Algorithm name contains '.filter' part, please rename it.\n"+
                                           " Use it only with filter names.");
        property string no_filter_part_in_filter: qsTr("Filter name does not contain part '.filter', please rename it.\n"+
                                           " Use it at the end of filter's name.");
        property string unavailable_name: qsTr("Unavailable name, please rename it.");

        objectName: "renameDiagramInfoDialog"
        title: qsTr("GalileoSkyScripter")
        text: default_name;
        buttons: StandardButton.Ok
        modality: Qt.ApplicationModal
        flags: dialogWindowFlags
    }

    MessageDialog {
        id: clearScriptDialog
        property string default_name: qsTr("Delete all scripts from the device?");

        objectName: "clearScriptDialog";
        title: qsTr("GalileoSkyScripter");
        text: default_name;
        buttons: StandardButton.Yes | StandardButton.No
        modality: Qt.ApplicationModal
        onYesClicked: {
            returnYes();
        }
        onNoClicked: {
            returnNo();
        }
        flags: dialogWindowFlags
    }

    MessageDialog {
        id: emptyDiagramInfoDialog
        objectName: "emptyDiagramInfoDialog"
        title: qsTr("GalileoSkyScripter")
        text: qsTr("Diagram name is empty.\n" +
                   "Specify diagram name, please.")
        buttons: StandardButton.Ok
        modality: Qt.ApplicationModal
        flags: dialogWindowFlags
    }

    MessageDialog {
        id: diagramErrorOnSave
        title: qsTr("GalileoSkyScripter")
        text: qsTr("Algorithm contains errors.\n" +
                   "Save only source file without binary?")
        buttons: StandardButton.Yes | StandardButton.No;
        modality: Qt.ApplicationModal
        onYesClicked: saveLocalRemote(false);
        flags: dialogWindowFlags
    }

    ParallelAnimation {
        id: animateDiagramCenter
        PropertyAnimation {
            id: animateX
            target: diagramField
            properties: "diagramCenterPointX"
            to: diagramCenterPointX
            easing.type: Easing.InOutQuad
            duration: 400
        }
        PropertyAnimation {
            id: animateY
            target: diagramField
            properties: "diagramCenterPointY"
            to: diagramCenterPointY
            easing.type: Easing.InOutQuad
            duration: 400
        }
    }
}
