import QtQuick 2.2
import QtQuick.Controls 1.2
import QtQuick.Controls.Styles 1.2
import QtQuick.Layouts 1.1
import QtQuick.Dialogs 1.2
import QtCharts 2.2
import com.galileosky.cppclasses 1.0

import "auxiliary.js" as Code
import "averageCalculation.js" as Avg

Rectangle {
    id: tarTableDialog
    objectName: "tarTableDialog"

    property var diagramItem: null
    property var parametersModel: ListModel{
        ListElement{ // пустой элемент для избежания ошибки при создании таймера.
            VarName: ""
        }
    }

    property int header_height: 20
    property int body_height: 370

    property int margin_offset : 5
    property int row_height: 20

    // Свойства для изменения положения диалога и его размера
    property real centerPointX: 0.0
    property real centerPointY: 0.0
    property real scaleFactorX: 0.0
    property real scaleFactorY: 0.0

    property var saveFileDialog: null;
    property int loaded_value_index: 0;
    property int loaded_value: 0;
    property int average_value: 0;
    property int pass_time: 0;
    property int measurment_count: 0;

    property string table_std_color: colorEditing
    property string table_y_std_color: "mediumorchid"
    property string table_active_color: colorSelected
    property string table_sustained_value_color: "lightgreen"
    property string table_gather_value_color: "gold"
    property string buttons_color: "white"
    property string std_transparent: "transparent"
    property string std_body: "white"
    property string std_popup: colorTextActionFinish
    property string mes_color: "white"

    property string second_title: ""

    property ListModel table_model: ListModel{
        ListElement{
            axis_x: qsTr("Sensor Value"); // Значение датчика
            axis_y: qsTr("Adjusted Value"); // Приведенное к единицам измерения
            is_start: "true"
        }

        onDataChanged: {
            console.debug("***** TABLE_MODEL CHANGED");
            changeLineSeries(table_model);
        }

        onCountChanged: {
            console.debug("***** TABLE_MODEL COUNT CHANGED");
            changeLineSeries(table_model);
        }

        onRowsInserted: {
            console.debug("***** TABLE_MODEL ROWS INDESRTED");
            changeLineSeries(table_model);
        }
    }

    width: 920
    height: header_height*2 + body_height + margin_offset*2
    radius: Math.round(10 * kDPI)

    color: colorDialogBackground
    border.width: 1
    border.color: colorDialogBorder
    z: zParameterDialog

    // Задает возможность перетаскивать диалоговое окно
    MouseArea {
        anchors.fill: parent
        drag.target: parent
        acceptedButtons: (Qt.LeftButton | Qt.RightButton)
        onPressed: {
            console.debug("z index:", tarTableDialog.z);
            thisDialogPressed(diagramItem)
        }
    }

    // Это самая важная штука для получения данных от прибора
    // Этот таймер дергает плюсовый объект, отвечающий за получение данных с прибора
    // Этот таймер запускается и отключается в нескольких местах в коде
    // Запуск - функция showItem() - там идет проверка установлена ли переменная от которой бдут полкчены данные.
    // Остановка - функция accept() - она вызывается при сокрытии окна диалгоа.
    Timer {
        id: load_val_timer
        interval: 1000;
        running: false
        repeat: true
        onTriggered: {
            ++pass_time;
            // попытка установить полученное значение
            setupRecievedValue();
            second_title = diagramItem.getConnectedVarName();
        }
    }

    // Функция установки диалога на первый план. Обработчик сигнала
    function setupZ(item){
        if (diagramItem === item) {
            tarTableDialog.z = zParameterDialog + 1;
            diagramItem.itemSelected(diagramItem, undefined);
            setupActiveFocusForTextinput();
        } else {
            tarTableDialog.z = zParameterDialog;
        }
    }
    // Сигнал отправляемый при нажатии на диалоговое окно
    signal thisDialogPressed(var item);

    Item{
       id: signals_handler
       visible: false;

       signal accept_value(int val);
       signal reset_popup();

       onAccept_value: {
           console.log("signals_handler: accept_value emitted");
       }
       onReset_popup: {
           console.log("signals_handler: reset_popup emitted");
       }
    }

    FocusScope {
        id: scope

        anchors.fill: parent
        //FocusScope needs to bind to visual properties of the children. Это из документации
        property alias color: scope_area.color

        onActiveFocusChanged: {
            if (scope.activeFocus){
                thisDialogPressed(diagramItem);
            }
        }

        Rectangle {
            id: scope_area
            anchors.fill: parent
            color: std_transparent
            Rectangle {
                id: main_container
                anchors.fill: parent
                anchors.leftMargin: margin_offset
                anchors.rightMargin: margin_offset
                anchors.topMargin: margin_offset
                anchors.bottomMargin: margin_offset

                Rectangle{
                    id: header
                    height: header_height;
                    width: parent.width
                    color: std_body

                    anchors.top: parent.top

                    Text{
                        font.bold: true
                        font.pixelSize: 12
                        text: qsTr("Calibration Table Dialog");
                    }

                    Rectangle {
                        id: closeRect
                        anchors.right: parent.right;
                        anchors.top: parent.top;
                        anchors.topMargin: 0
                        anchors.rightMargin: 0
                        width: Math.round(18 * kDPI)
                        height: Math.round(18 * kDPI)

                        Image {
                            id: closeImage

                            source: "images/icon_close.svg"
                            sourceSize.width: Math.round(18 * kDPI)
                            sourceSize.height: Math.round(18 * kDPI)
                            smooth: false

                            MouseArea {
                                id: closeRectArea
                                objectName: "closeRectArea"
                                anchors.fill: parent
                                onClicked: {
                                    accept();
                                    // уничтожение окна диалога.
                                    // Забавно но эта функция не подставляется системой,
                                    // но она определена в классе Rectangle, ну скорее Item.
                                    diagramItem.is_dialog_created = false;
                                    forceActiveFocus();
                                    diagramItem.notWantAnymore();
                                    tarTableDialog.destroy();
                                }
                            }
                        }
                    }
                }

                Rectangle{
                    id: underheader
                    height: header_height;
                    width: parent.width
                    color: std_body
                    anchors.top: header.bottom

                    Text{
                        anchors.left: parent.left;
                        anchors.leftMargin: 20;
                        anchors.top: parent.top
                        anchors.topMargin: 3;
                        font.pixelSize: 10
                        text: second_title;
                    }
                }


                Rectangle{
                    id: body
                    height: body_height;
                    width: parent.width
                    color: std_body

                    anchors.top: underheader.bottom


                    Rectangle{
                        id: calibration_table_container
                        width: calibration_table.width
//                        height: 300 //Math.round(300*kDPI);

                        anchors.top: parent.top
                        anchors.topMargin: margin_offset*2

                        anchors.left: parent.left
                        anchors.leftMargin: margin_offset*2

                        anchors.bottom: table_border.bottom
                        anchors.bottomMargin: margin_offset*2

                        color: std_transparent

                        Rectangle{
                            id: calibration_table

                            property int cell_width: 110
                            property int rem_rect_width: 20
                            property int scroller_width: 10
                            property int popup_width: 235

                            width: (cell_width*2) + rem_rect_width + scroller_width + popup_width;
                            height: parent.height

                            anchors.top: parent.top
                            anchors.right: parent.right

                            color: std_transparent

                            ScrollView {
                                anchors.fill: parent

                                style: MyScrollViewStyle {}
                                ListView {
                                    id: lv
                                    model: table_model
                                    delegate: Item {
                                        id: values_row
                                        width: parent.width
                                        height: is_remove_visible ? row_height : head_heigth

                                        property string text_x: model.axis_x
                                        property string text_y: model.axis_y
                                        property int head_heigth: row_height*(1.5)
                                        property bool is_remove_visible: !(model.is_start === "true");

                                        Rectangle{
                                            id: popup_rect;
                                            objectName: "Popup"
                                            width: calibration_table.popup_width;
                                            height: 2*row_height;

                                            visible: false
                                            anchors.top: parent.top
                                            anchors.topMargin: -row_height;
                                            color: std_transparent;

                                            Rectangle{
                                                width: parent.width - 8;
                                                height: parent.height;
                                                color: std_popup;
                                            }
                                            Rectangle{
                                                width: row_height/Math.sqrt(2);
                                                height: row_height/Math.sqrt(2);
                                                color: std_popup;
                                                rotation: 45;
                                                anchors.bottom: parent.bottom;
                                                anchors.bottomMargin: 3;
                                                anchors.right: parent.right;
                                                anchors.rightMargin: 3;
                                            }

                                            Loader {
                                                id: popup_cloud;
                                                anchors.top: parent.top
                                                anchors.topMargin: row_height;
                                            }
                                        }

                                        Rectangle{
                                            id: axis_x
                                            width: calibration_table.cell_width
                                            height: is_remove_visible ? row_height : head_heigth
                                            color: table_std_color
                                            border.width: 1
                                            border.color: colorDialogBorder
                                            anchors.left: popup_rect.right;                                            
                                            TextInput{
                                                id: text_axis_x
                                                width: parent.width
                                                height: parent.height
                                                horizontalAlignment: TextEdit.AlignHCenter
                                                verticalAlignment: TextEdit.AlignVCenter
                                                color: "white"
                                                font.pixelSize: is_remove_visible ? 15 : 12
                                                font.bold: true
                                                text: text_x
                                                readOnly: !is_remove_visible
                                                wrapMode: TextEdit.WordWrap
                                                validator: RegExpValidator { regExp:/[0-9]+/ }
                                                Keys.onEnterPressed: {
                                                    // Перевод фокуса на воторой элемент в строке таблицы
                                                    setFocusToYCell(text_axis_x, text_axis_y, event.key);
                                                }
                                                Keys.onReturnPressed: {
                                                    // Перевод фокуса на воторой элемент в строке таблицы
                                                   setFocusToYCell(text_axis_x, text_axis_y, event.key);
                                                }
                                                onFocusChanged: {
                                                    if (true === focus) {
                                                        // сразу выделим все чтобы было удобно редактировать
                                                        lv.currentIndex = model.index;
                                                        // при этом у нас фокус может слететь(из-за игра с currentIndex), поэтому просто еще раз присваиваем фокус этому элементу.
                                                        if (is_remove_visible) {
                                                            parent.color = table_active_color;
                                                            selectAll();
                                                            focus = true
                                                            signals_handler.accept_value.connect(acceptValue);
                                                            signals_handler.reset_popup.connect(resetPopup);

                                                            popup_cloud.sourceComponent = value_sower;
                                                            popup_rect.visible = true;
                                                            resetAvgValue();
                                                        }
                                                    } else {
                                                        parent.color = table_std_color;
                                                        // принимаем последенее значение TextEdit
                                                        acceptXValue(model.index, text);
                                                        deselect();
                                                        popup_cloud.sourceComponent = null;
                                                        popup_rect.visible = false;
                                                        signals_handler.accept_value.disconnect(acceptValue);
                                                        signals_handler.reset_popup.disconnect(resetPopup);
                                                    }
                                                }
                                                // Слот для сигнала.
                                                function acceptValue(val){
                                                    var new_text = "%1".arg(val);
                                                    // Выводим текст в ячейку
                                                    text = new_text;
                                                    // Записываем текст в модель
                                                    acceptXValue(model.index, new_text);
                                                    // выделяем новый текст чтобы можно было его начать редактировать
                                                    resetPopup();
                                                }
                                                function resetPopup(){
                                                    // выделяем новый текст чтобы можно было его начать редактировать
                                                    text_axis_x.forceActiveFocus();
                                                    selectAll();
                                                }
                                            }
                                        }
                                        Rectangle {
                                            id: axis_y
                                            width: calibration_table.cell_width
                                            height: is_remove_visible ? row_height : head_heigth
                                            color: table_std_color
                                            anchors.left: axis_x.right;
                                            border.width: 1
                                            border.color: colorDialogBorder                                            
                                            TextInput{
                                                id: text_axis_y
                                                horizontalAlignment: TextEdit.AlignHCenter
                                                verticalAlignment: TextEdit.AlignVCenter
                                                width: parent.width
                                                height: parent.height
                                                color: "white"
                                                font.pixelSize: is_remove_visible ? 15 : 12
                                                font.bold: true
                                                text: text_y
                                                readOnly: !is_remove_visible
                                                wrapMode: TextEdit.WordWrap
                                                validator: RegExpValidator { regExp:/[0-9]+/ }
                                                Keys.onEnterPressed: {
                                                    handleYreturnPressed(event.key, text_axis_y);
                                                }
                                                Keys.onReturnPressed: {
                                                    handleYreturnPressed(event.key, text_axis_y);
                                                }
                                                onFocusChanged: {
                                                    if (true === focus) {
                                                        // Здесь идея такая, что нужно установить текущее значение активной строки в ListView
                                                        lv.currentIndex = model.index;
                                                        // при этом у нас фокус может слететь(из-за игра с currentIndex), поэтому просто еще раз присваиваем фокус этому элементу.
                                                        focus = true;
                                                        // Если у нас не заголовок, то мы можем получить значение с выбранной переменноой, сделаем видимой кнопку для этого.
                                                        if (is_remove_visible) {
                                                            parent.color = table_active_color;
                                                            // И сразу выделим текст чтобы его было удобно редактировать
                                                            selectAll();
                                                            resetAvgValue();
                                                        }
                                                    } else {
                                                        parent.color = table_std_color;
                                                        // Принимаем значение которое осталось в textEdit
                                                        acceptYValue(model.index, text);
                                                        // скрываем кнопку применения значения из переменной
                                                        deselect();
                                                    }
                                                }
                                            }
                                        }
                                        Rectangle {
                                            id: removeRect
                                            width: Math.round(18 * kDPI)
                                            height: Math.round(18 * kDPI)
                                            color: std_transparent
                                            anchors.left: axis_y.right;

                                            Image {
                                                id: removeImage

                                                visible: is_remove_visible
                                                source: "images/icon_close.svg"
                                                sourceSize.width: Math.round(18 * kDPI)
                                                sourceSize.height: Math.round(18 * kDPI)
                                                smooth: false

                                                MouseArea {
                                                    id: removeRectArea
                                                    objectName: "closeRectArea"
                                                    anchors.fill: parent
                                                    onClicked: {
                                                        console.debug("tarTableDialog. Calibration table. remove data")
                                                        forceActiveFocus();
                                                        removeCalibrationData(model.index);
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }

                    Rectangle{
                        id: plot_container
                        width: 400
                        height: 360

                        color: std_transparent

                        anchors.top: parent.top
                        anchors.topMargin: margin_offset*2

                        anchors.right: parent.right
                        anchors.rightMargin: margin_offset*2

                        Rectangle{
                            id: plot

                            width: 400
                            height: 300

                            Loader{
                                anchors.fill: parent
                                sourceComponent: chart_view
                            }

                            ChartView {
                                id: view
                                anchors.fill: parent
                                antialiasing: true

                                AreaSeries {
                                    id: areaSer
                                    name: qsTr("Interpolation of Table");
                                    color: std_transparent
                                    borderColor: "#FF0039A5"
                                    borderWidth: 1

                                    axisX: ValueAxis {
                                        id: valueAxisX
                                        min: 0
                                        max: 100
                                        tickCount: 6
                                        labelFormat: "%.0f"
                                    }
                                    axisY: ValueAxis {
                                        id: valueAxisY
                                        min: 0
                                        max: 100
                                        tickCount: 6
                                        labelFormat: "%.0f"
                                    }
                                    upperSeries: LineSeries {
                                        id: plot_line
                                    }
                                }
                            }
                        }
                    }

                    Rectangle {
                        id: table_border
                        color: std_transparent;

                        width: ((calibration_table.cell_width*2) + (calibration_table.rem_rect_width) + calibration_table.scroller_width)//calibration_table.cell_width*2 + 2*margin_offset;

                        anchors.top: parent.top;
                        anchors.topMargin: margin_offset*2;

                        anchors.left: parent.left
                        anchors.leftMargin: calibration_table.popup_width + margin_offset*2;

                        anchors.bottom: add_slot_table.top
                        anchors.bottomMargin:margin_offset

                        border.width: 1
                        border.color: colorDialogBorder
                    }

                    Button {
                        anchors.left: table_border.left
                        anchors.leftMargin: margin_offset

                        anchors.bottom: load_table.top
                        anchors.bottomMargin: margin_offset

                        id: add_slot_table
                        text:  qsTr("Add values")
                        style: ButtonStyle {
                            background: Rectangle {
                                implicitWidth: table_border.width - margin_offset*2
                                implicitHeight: row_height
                                radius: row_height/4
                                border.width: 1
                                border.color: "#888"
                                gradient: Gradient {
                                    GradientStop { position: 0 ; color: control.pressed ? "#ccc" : "#eee" }
                                    GradientStop { position: 1 ; color: control.pressed ? "#aaa" : "#ccc" }
                                }
                            }
                        }
                        onClicked: {
                            forceActiveFocus();
                            focus = true;
                            addNewCalibratinData();
                        }
                    }

                    Text {
                        id: save_btt_txt
                        text: qsTr("Save Table")
                        anchors.centerIn: parent;
                        visible: false
                    }

                    Button {
                        anchors.bottom: body.bottom
                        anchors.bottomMargin: margin_offset
                        anchors.left: body.left
                        anchors.leftMargin: margin_offset

                        id: save_table
                        text:  save_btt_txt.text
                        style: ButtonStyle {
                            background: Rectangle {
                                implicitWidth: save_btt_txt.width + 20
                                implicitHeight: row_height
                                radius: row_height/4
                                border.width: 1
                                border.color: "#888"
                                gradient: Gradient {
                                    GradientStop { position: 0 ; color: control.pressed ? "#ccc" : "#eee" }
                                    GradientStop { position: 1 ; color: control.pressed ? "#aaa" : "#ccc" }
                                }
                            }
                        }
                        onClicked: {
                           forceActiveFocus();
                           focus = true;
                           var name = fileIOQML.getSaveFileName();
                           var data = serializeTableDataToCsv(table_model);
                           saveCalTableToFile(name, data);
                        }
                    }

                    Text {
                        id: load_btt_txt
                        text: qsTr("Load Table");
                        anchors.centerIn: parent;
                        visible: false
                    }
                    Button {
                        anchors.bottom: body.bottom
                        anchors.bottomMargin: margin_offset
                        anchors.left: save_table.right
                        anchors.leftMargin: margin_offset*2

                        id: load_table
                        text:  load_btt_txt.text
                        style: ButtonStyle {
                            background: Rectangle {
                                implicitWidth: load_btt_txt.width + 20
                                implicitHeight: row_height
                                radius: row_height/4
                                border.width: 1
                                border.color: "#888"
                                gradient: Gradient {
                                    GradientStop { position: 0 ; color: control.pressed ? "#ccc" : "#eee" }
                                    GradientStop { position: 1 ; color: control.pressed ? "#aaa" : "#ccc" }
                                }
                            }
                        }
                        onClicked: {
                            forceActiveFocus();
                            var name = fileIOQML.getOpenFileName();
                            if ("" !== name) {
                                var data = loadCalTableFromFile(name);
                                deserializeFromCsv(data);
                            }
                        }
                    }
                }
            }
        }
    }


    function dump() {
        console.debug("---- tarTableDialog dump start:", objectName);
        var i;
        console.debug("parametersModel:", parametersModel);
        if (parametersModel !== null) {
            for (i = 0; i < parametersModel.count; ++i)
                console.debug(parametersModel.get(i).paramName,
                              parametersModel.get(i).paramType,
                              parametersModel.get(i).varName);
        }        
        console.debug("---- tarTableDialog dump finish:", objectName);

    }
    function tryToStartTimer(){
        var is_var_selected = (parametersModel.get(0).varName !== "");
        if (is_var_selected) {
            diagramItem.want();
            load_val_timer.start();
        }
    }

    function showItem(item) {
        diagramItem = item

        tarTableDialog.parametersModel = item.parametersModel;

        // Теперь проверим данные в новой модели. И запишем все что используем внутри.
        var count = parametersModel.count;

        // Заполним таблицу
        var first_element = Code.clone(table_model.get(0));
        table_model.clear();
        table_model.append(first_element);
        // Дальше если есть данные о таблице - записываем их
        for (var i = 2; i < count-1; ++i) {
            var element_x = parametersModel.get(i);
            var element_y = parametersModel.get(++i);
            var new_element = {"axis_x":element_x.varName, "axis_y":element_y.varName}
            table_model.append(new_element);
        }

        dump();

        // Теперь попробуем определить индекс вычисляемого параметраи установить команду для него
        tryToStartTimer();
        diagramItem.newParameterSetted.connect(tryToStartTimer);

        // center dialog to item
        var itemCenter = diagramContainer.mapFromItem(item, item.width/2, item.height/2);
        centerPointX = itemCenter.x;
        centerPointY = itemCenter.y;
        // set size to item and show,...
        scaleFactorX = diagramContainer.diagramField.itemWidth / width;
        scaleFactorY = diagramContainer.diagramField.itemHeight / height;
        // ...than animate to normal size
        centerPointX = parent.width/2
        centerPointY = parent.height/2
        scaleFactorX = 1.0;
        scaleFactorY = 1.0;
    }

    function clearParametersModel(){
        var tar_table_model_parameter_start_pos = 2; // Это позиция последнего стандартного элемента в модели
        var count = parametersModel.count;
        var std_count = tar_table_model_parameter_start_pos + 1;
        if (count > std_count) {
            var amount = count - std_count;
            parametersModel.remove(tar_table_model_parameter_start_pos, amount);
        }
    }

    function accept() {
        console.debug(objectName, arguments.callee.name);
        acceptLastModified();

        clearParametersModel();
        // Заполняем модель значениями таблицы.
        var count = table_model.count;
        var local_tar_table = [];
        for (var i = 1; i< count; ++i) {

            var axis_x_name = table_model.get(i).axis_x;
            var axis_y_name = table_model.get(i).axis_y;

            var axis_x_var = {"varName":axis_x_name, "paramType":"int", "varCategory": "const", "varBaseCategory": categoryConst};
            var axis_y_var = {"varName":axis_y_name, "paramType":"int", "varCategory": "const", "varBaseCategory": categoryConst};

            // Добавляем в модель данных диалога данные таблицы тарировки
            parametersModel.insert(parametersModel.count-1, axis_x_var);
            parametersModel.insert(parametersModel.count-1, axis_y_var);
            local_tar_table.push(parseInt(axis_x_name));
            local_tar_table.push(parseInt(axis_y_name));

            // Так же добавим эти переменные в модель переменных, чтобы они потом могли быть восстановлены
            // И чтобы мы потом не придумывали отдельный тип для них.
            if (varsModel.getVarId(axis_x_name, categoryConst) === -1) {
                varsModel.addVar(categoryConst, "int","Algorithms", axis_x_name /*name*/,""/*description*/, axis_x_name /*value*/);
            }
            if (varsModel.getVarId(axis_y_name, categoryConst) === -1) {
                varsModel.addVar(categoryConst, "int","Algorithms", axis_y_name /*name*/,""/*description*/, axis_y_name /*value*/);
            }
        }
        diagramItem.tar_table = Code.clone(local_tar_table);
    }

    // Эта функция вызывается когда выделяется объект на поле диаграммы
    function selectionChanged(item, mouseEvent, itemWasSelected) {
        console.debug(objectName, "selectionChanged()", item)
        if ((visible === false) || (item === tarTableDialog))
            return;

        accept();
        dump();
    }

    function addNewCalibratinData() {
        var param_count = table_model.count;
        table_model.insert(param_count, {"axis_x":"0", "axis_y":"0"})
    }

    function removeCalibrationData(index) {
        table_model.remove(index);
    }

    function setFocusToYCell(item1, item2, key_event){
        item1.focus = false
        item2.focus = true;
        item2.forceActiveFocus();
        item2.selectAll();
    }

    function saveCalTableToFile(file_name, data){
        fileIOQML.write(file_name, data);
    }

    function loadCalTableFromFile(file_name) {
        var data = fileIOQML.load(file_name);
        return data;
    }

    function serializeTableDataToCsv(table_model){
        var data = "";
        var i;
        for (i = 1; i < table_model.count; ++i) {
          var x_str = table_model.get(i).axis_x;
          var y_str = table_model.get(i).axis_y;
          x_str.replace("\n","");
          y_str.replace("\n","");
          x_str.replace("\r","");
          y_str.replace("\r","");

          var end_line = "\n";
          data += x_str + ";" + y_str + end_line;
        }
        return data;
    }

    function deserializeFromCsv(data) {
        data = data.replace(/\r/g,"");
        data = data.replace(/\n+/g,"\n");
        var rows = data.split("\n");

        if (table_model.count !== 1){
            table_model.remove(1, table_model.count - 1);
        }

        var counter = 0;
        var i;
        for (i = 0; i < rows.length; ++i) {
            var cols = rows[i].split(";");
            var x_str = cols[0];
            var y_str = cols[1];            
            if (x_str !=="" && y_str !=="" ) {
                console.debug("cols :", cols, y_str.indexOf("\n"));
                x_str.replace(/\n/gm,"_");
                y_str.replace(/\n/gm,"_");
                table_model.append({"axis_x": x_str, "axis_y": y_str});
            }
        }
    }

    function acceptXValue(index, value){
        table_model.set(index, {"axis_x": value})
    }
    function acceptYValue(index, value){
        table_model.set(index, {"axis_y": value})
    }

    function changeLineSeries(data_model){
        if (data_model.count === 1) {
            return;
        }

        plot_line.clear();

        var x_data =[];
        var y_data =[];

        var i;
        for (i = 1; i < data_model.count; ++i) {
            var tx = data_model.get(i).axis_x;
            var ty = data_model.get(i).axis_y;
            // Нужно загружать только полные поля
            console.debug("model data x =", tx,"y =", ty);
            if (tx !=="" && ty !=="" ) {
                x_data.push(parseFloat(tx));
                y_data.push(parseFloat(ty));
            }
        }

        var sorted_x_data = [];
        for (i = 0; i < x_data.length; ++i) {
            sorted_x_data.push(x_data[i]);
        }

        var sorted_y_data = [];

        sorted_x_data.sort(compareNumericMinFirst);
        for (i = 0; i < sorted_x_data.length; ++i) {
            var comp_val = sorted_x_data[i];
            for (var j = 0; j < y_data.length; ++j) {
                if (comp_val === x_data[j]) {
                    sorted_y_data[i] = y_data[j] ;
                }
            }
        }

        for (i = 0; i < sorted_x_data.length; ++i) {
            var x = sorted_x_data[i];
            var y = sorted_y_data[i];
            plot_line.append(x, y);
        }
        if (sorted_x_data.length < 2) {
            return;
        }

        view.axisX(areaSer).min = sorted_x_data[0];
        view.axisX(areaSer).max = sorted_x_data[sorted_x_data.length-1];

        sorted_y_data.sort(compareNumericMinFirst);
        view.axisY(areaSer).min = sorted_y_data[0];
        view.axisY(areaSer).max = sorted_y_data[sorted_y_data.length-1];

        areaSer.lowerSeries = areaSer.upperSeries;
    }

    // Функции определения граничных значений для графика
    function setupBorders(plot_data){
        var model_data = plot_data;
        // Мы рисуем график.

        //0. Подготовим данные для удобного использования
        var x_data =[];
        var y_data =[];
        var i;
        for (i = 0; i < model_data.count; ++i) {
            x_data.push(model_data.get(i).axis_x);
            y_data.push(model_data.get(i).axis_y);
        }

//        1. Определимся с размером оси Х
        var max_x = getMax(x_data);
        var min_x = getMin(x_data);
//        2. Определимся с размером оси У
        var max_y = getMax(y_data);
        var min_y = getMin(y_data);

//       3. Запишем новые границы
        valueAxisX.min = min_x;
        valueAxisX.max = max_x;

        valueAxisY.min = min_y;
        valueAxisY.max = max_y;
    }

    function getMax(data_array){
        var tmp_data = data_array;
        tmp_data.sort(compareNumericMaxFirst);
        return tmp_data[0];
    }

    function compareNumericMaxFirst(a, b) {
      if (a > b) return -1;
      if (a < b) return 1;
    }

    function getMin(data_array){
        var tmp_data = data_array;
        tmp_data.sort(compareNumericMinFirst);
        return tmp_data[0];
    }

    function compareNumericMinFirst(a, b) {
      if (a > b) return 1;
      if (a < b) return -1;
    }

    function setupParameterID(param_name){
        var k;
        for (k = 0; k < localConstVarsProxyModel.count; ++k) {
          if (localConstVarsProxyModel.get(k, "varName") === param_name) {
            loaded_value_index = k;
            return true;
          }
        }
        return false;
    }

    function setupRecievedValue(){
        if (diagramItem.is_received_value_new) {
            loaded_value = diagramItem.getReceivedValue();
            Avg.addAvgData(loaded_value);
            measurment_count = Avg.getMeasurementCount();
            average_value = Avg.evalM(measurment_count);
        }
    }

    function handleYreturnPressed(event_key, text_axis_y) {
        text_axis_y.focus = false;
        // Если активен последний элемент представления, мы не переходим на следующий
        if (lv.currentIndex === lv.count-1){
            return;
        }
        // переход на следующий элемент пердставления.
        lv.incrementCurrentIndex();
        // Row         Rectangle   TextEdit
        lv.currentItem.children[1].children[0].focus = true;
    }

    function acceptLastModified(){
        // Для того чтобы принять все модифицированные данные нужно просто по закрытии
        // изменить фокус, что вызовет ринятие данных.
        // Чтобы не выбирать просто последний выделенный выберем а потом снимем с него фокус, тогда все должно отработать.
        // Хотя все еще немного хитрее у нас фокус может быть у 2-х элементов в текущем индексе.
                             // Row         Rectangle   TextEdit
        var axis_x_element = lv.currentItem.children[1].children[0];
        var axis_y_element = lv.currentItem.children[2].children[0];

        // Теперь у нас поведение изменилось - нужно снова фокус этому элементу вернуть
        // Тогда мы применим изменения и у нас окно всплывающее не пропадет.
        if (axis_x_element.focus) {
            axis_x_element.focus = false;
            axis_x_element.focus = true;
        }
        if (axis_y_element.focus) {
            axis_y_element.focus = false;
            axis_y_element.focus = true;
        }
    }

    function setupActiveFocusForTextinput(item) {
        // Такой вызов обеспечивает редактирование внутри диалогового окна при его выборе
        scope.forceActiveFocus();
    }

    function resetAvgValue(){
        Avg.clearAvgData();
        pass_time = 0;
        average_value = 0;
        measurment_count = 0;
        loaded_value = 0;
    }

    Component {
        id: chart_view
        ChartView {
            anchors.fill: parent
            antialiasing: true

            AreaSeries {
                name: "Russian"
                color: "#FFD52B1E"
                borderColor: "#FF0039A5"
                borderWidth: 1
                axisX: ValueAxis {
                    id: valueAxisX
                    min: 0
                    max: 100
                    tickCount: 50
                    labelFormat: "%.0f"
                }
                axisY: ValueAxis {
                    id: valueAxisY
                    min: 0
                    max: 100
                    tickCount: 50
                    labelFormat: "%.0f"
                }
                upperSeries: LineSeries {
                    id: plot_line
                }
            }
        }
    }

    Component {
        id: value_sower


        Rectangle{
            id: value_sower_rect

            color: std_transparent
            width: 300
            height: row_height
                Rectangle{
                    id: acc_rect
                    width: 100
                    height: row_height
                    radius: row_height/4

                    color: buttons_color
                    border.width: 1
                    border.color: colorDialogBorder

                    Text {
                        id: value_txt
                        text: qsTr("Current: %1").arg(loaded_value);
                        anchors.centerIn: parent;
                    }

                    MouseArea {
                        id: accept_value_area
                        objectName: "acceptValueArea"
                        anchors.fill: parent
                        onClicked: {
                            console.debug("accept_value_area. accept clicked");
                            signals_handler.accept_value(loaded_value);
                        }
                    }
                }

                Rectangle{
                    id: mes_and_time_rect
                    width: 200
                    height: row_height

                    color: std_transparent

                    anchors.left: parent.left
                    anchors.bottom: parent.top

                    Text {
                        id: mes_and_time_txt
                        color: mes_color
                        text: qsTr("%1 measurments, by %2 sec").arg(measurment_count).arg(pass_time);
                        anchors.centerIn: parent;
                    }
                }

                Rectangle{
                    id: acc_avg_rect
                    width: 100
                    height: row_height
                    radius: row_height/4

                    color: buttons_color
                    border.width: 1
                    border.color: colorDialogBorder

                    anchors.left: acc_rect.right

                    Text {
                        id: avg_value_txt
                        text: qsTr("Average: %1").arg(average_value);
                        anchors.centerIn: parent;
                    }

                    MouseArea {
                        id: accept_avg_value_area
                        objectName: "acceptValueArea"
                        anchors.fill: parent
                        onClicked: {
                            console.debug("accept_avg_value_area. accept clicked");
                            signals_handler.accept_value(average_value);
                        }
                    }
                }

                Rectangle{
                width: row_height
                height: row_height
                radius: row_height/4

                color: buttons_color
                border.width: 1
                border.color: colorDialogBorder

                anchors.left: acc_avg_rect.right

                Text {
                    id: reset_avg_value_txt
                    font.pixelSize: row_height - 2;
                    text: "\u27F3"// Символ стрелки обновления
                    anchors.bottom: parent.bottom
                    anchors.right: parent.right;
                    anchors.rightMargin: 2
                }

                MouseArea {
                    id: reset_avg_value_area
                    objectName: "acceptValueArea"
                    anchors.fill: parent
                    onClicked: {
                        resetAvgValue();
                        signals_handler.reset_popup();
                    }
                }
            }
        }
    }

    transform: [
        Scale {
            origin.x: width/2
            origin.y: height/2
            xScale: scaleFactorX
            yScale: scaleFactorY
        },
        Translate {
            x: centerPointX - width/2
            y: centerPointY - height/2
        }
    ]
    Behavior on centerPointX {
        enabled: visible
        NumberAnimation {
            easing.type: Easing.Linear
        }
    }
    Behavior on centerPointY {
        enabled: visible
        NumberAnimation {
            easing.type: Easing.Linear
        }
    }

    Component.onCompleted: {
        diagramField.selectionChanged.connect(tarTableDialog.selectionChanged)
        diagramContainer.setupZ.connect(tarTableDialog.setupZ);
        console.debug(objectName, "completed")
    }
    Component.onDestruction: {
        diagramField.selectionChanged.disconnect(tarTableDialog.selectionChanged)
        diagramContainer.setupZ.disconnect(tarTableDialog.setupZ);
        console.debug(objectName, "destructed")
    }
}

