import QtQuick 2.0

Item {
    id: diagramLine
    objectName: "diagramLine"

    property bool created: false
    property point endpoint1 : "0,0"
    property point endpoint2 : "0,0"
    property string point1Type : "down"     // right, left, up
    property var middlePoints : []
    property int pageIndex      // page of the diagram which contains this

    property color fillStyle:   colorDiagramLine
    property color strokeStyle: fillStyle
    property int lineWidth: regularThickness

    // Добавим хранение объектов в которым привязана линия, а не просто установку ее координат.
    // такой подход даст возсоженосить определять какая линия должна сществовать а какая нет когда работаем с
    // несколькими точками подцепления.
    property var startPoint: null
    property var endPoint: null

    property string prevState: ""   // the state before hovering
    property bool is_filter: false

    width: 16
    height: 16
    z: (pageIndex === diagramContainer.pageIndex) ? zDiagramLine : zDiagramObjectOnHiddenPage
    enabled: pageIndex === diagramContainer.pageIndex

    signal lineIsAboutToDestroy(Item item)

    function dump() {
        console.debug("---- dump:",
                    objectName, pageIndex, endpoint1, endpoint2, middlePoints);
    }

    function isLineSegmentClicked(endpoint1, endpoint2, x, y) {
        var isBetweenPoints = false;
        var clickableDistance = 4;

        if (((Math.min(endpoint1.x, endpoint2.x)-clickableDistance) < x)
                && (x < (Math.max(endpoint1.x, endpoint2.x)+clickableDistance))
                && ((Math.min(endpoint1.y, endpoint2.y)-clickableDistance) < y)
                && (y < (Math.max(endpoint1.y, endpoint2.y)+clickableDistance)))
        {
            isBetweenPoints = true;
        }

        var distanceToLine =
                Math.abs(((endpoint2.x-endpoint1.x)*(endpoint1.y-y)
                          - (endpoint1.x-x)*(endpoint2.y-endpoint1.y)))
                / Math.sqrt(Math.pow(endpoint2.x-endpoint1.x, 2)
                            + Math.pow(endpoint2.y-endpoint1.y, 2));

        return isBetweenPoints && (distanceToLine < clickableDistance*2)
    }

    function isLineClicked(x, y) {
        if (isLineSegmentClicked(endpoint1, middlePoints[0], x, y))
            return true;

        var i;
        for (i = 0; i< middlePoints.length - 1; ++i)
            if (isLineSegmentClicked(middlePoints[i], middlePoints[i+1], x, y))
                return true;

        return isLineSegmentClicked(middlePoints[i], endpoint2, x, y);
    }

    function selectionChanged(item, mouseEvent, itemWasSelected) {
        console.debug(objectName, "selectionChanged()", item)

        if (pageIndex !== diagramContainer.pageIndex)
            return;

        // select self
        if (item === diagramLine) {
            diagramLine.state = "selected"
            diagramLine.prevState = diagramLine.state;
            return
        }

        // Ctrl is pressed
        if ((mouseEvent !== undefined)
                && (mouseEvent.modifiers & Qt.ControlModifier))
        {
            // invert state on hit
            if (isLineClicked(mouseEvent.x, mouseEvent.y)) {
                diagramLine.state =
                        ((diagramLine.state === "selected")
                         || (diagramLine.prevState === "selected"))
                        ? "" : "selected";
            }
            // ... no changes otherwise
            else {
                ;
            }
        }
        // Ctrl isn't pressed
        else
        {
            // select on hit
            if ((mouseEvent !== undefined)
                && isLineClicked(mouseEvent.x, mouseEvent.y))
            {
                diagramLine.state = "selected"
            }
            // ... deselect otherwise
            else {
                diagramLine.state = ""
            }
        }
        diagramLine.prevState = diagramLine.state;
    }

    function destroyItem() {
        lineIsAboutToDestroy(diagramLine);
        destroy();
    }

    function lineEntered(linePrimitive) {
        prevState = state;
        state = "hovered";
    }
    function lineExited(linePrimitive) {
        state = prevState;
    }

    // set Endpoint1 coordinates according to \a diagramObject \a type
    function setEndpoint1(type, diagramObject) {
         // Если фильтр, делаем присоединение сбоку и выходим.
        console.debug("***** SetEndpoint 1");
        startPoint = diagramObject;
        if ("filter" === diagramObject.filter_type) {
            is_filter = true;
            endpoint1.x = diagramObject.x + diagramObject.width - diagramObject.itemRectMargin;
            endpoint1.y = diagramObject.y + diagramObject.height / 2;
            middlePoints.push(Qt.point(endpoint1.x, endpoint1.y));
            return;
        }
        point1Type = type;
        if (point1Type === "right") {
            endpoint1.x = diagramObject.x + diagramObject.width - diagramObject.itemRectMargin;
            endpoint1.y = diagramObject.y + diagramObject.height / 2;
            middlePoints.push(Qt.point(endpoint1.x+diagramField.gridSize, endpoint1.y));
        }
        if (point1Type === "left") {
            endpoint1.x = diagramObject.x + diagramObject.itemRectMargin;
            endpoint1.y = diagramObject.y + diagramObject.height / 2;
            middlePoints.push(Qt.point(endpoint1.x-diagramField.gridSize, endpoint1.y));
        }
        else if (point1Type === "down") {
            endpoint1.x = diagramObject.x + diagramObject.width / 2;
            endpoint1.y = diagramObject.y + diagramObject.height - diagramObject.itemRectMargin;
            middlePoints.push(Qt.point(endpoint1.x, endpoint1.y+diagramField.gridSize));
        }
    }

    function setEndpoint2(diagramObject) {
        console.debug("***** SetEndpoint 2");
        // Если фильтр, делаем присоединение сбоку и выходим.
        endPoint = diagramObject;
        if ("filter" === diagramObject.filter_type) {
            is_filter = true;
            var conn_x = diagramObject.connect_pos_x;
            var conn_y = diagramObject.connect_pos_y;
            var two_connection_filter_type = 4; // типа const, не менять на ходу.
            if (two_connection_filter_type === diagramObject.view_type) {
                var tmp_obj = diagramObject.getPortByItem(startPoint);
                if (null !== tmp_obj) {
                    conn_x = tmp_obj.connect_pos_x
                    conn_y = tmp_obj.connect_pos_y
                }
            }

            endpoint2.x = conn_x;
            endpoint2.y = conn_y;  /*+ diagramObject.actual_height/ 2*///+ diagramObject.itemRectMargin;
            middlePoints.push(Qt.point(endpoint2.x-diagramField.gridSize, endpoint2.y));
            console.log(diagramObject);
            console.log("ep2_x:", conn_x, "ep2_y:", conn_y, "act_y_pos:", diagramObject.y, " ___ ", diagramObject.objectName);
            return;
        }
        endpoint2.x = diagramObject.x + diagramObject.width / 2;
        endpoint2.y = diagramObject.y + diagramObject.itemRectMargin;
        middlePoints.push(Qt.point(endpoint2.x, endpoint2.y-diagramField.gridSize));
    }

    function createPrimitive(point1, point2) {
        console.debug("createPrimitive():", point1, "<->", point2);
        var x1 = Math.min(point1.x, point2.x);
        var x2 = Math.max(point1.x, point2.x);
        var y1 = Math.min(point1.y, point2.y);
        var y2 = Math.max(point1.y, point2.y);

        // load primitive file
        var component = Qt.createComponent("DiagramLinePrimitive.qml");
        console.debug("component.status:", component.status, "\t",
                      "component.errorString:", component.errorString());
        // create primitive object
        var newObject = component.createObject(diagramLine);
        if (newObject === null) {
            console.warn("createPrimitive() createObject() : error");
            return;
        }

        // set primitive properties
        newObject.objectName =
                'lineSegmentRect_'+diagramLine.children.length;
        newObject.actualX = x1;
        newObject.actualY = y1;
        newObject.actualThickness = Qt.binding(function() {
            return diagramLine.lineWidth;
        });
        newObject.color = Qt.binding(function() {
            return diagramLine.fillStyle;
        });
        // choose line orientation
        if (x1 === x2) {
            newObject.orientation = Qt.Vertical;
            newObject.actualLength = y2 - y1;
        }
        else {
            newObject.orientation = Qt.Horizontal;
            newObject.actualLength = x2 - x1;
        }
        newObject.lineArea.entered.connect(lineEntered);
        newObject.lineArea.exited.connect(lineExited);
        console.debug(newObject, ":", newObject.objectName, ":",
                      newObject.x, newObject.y,
                      newObject.width, newObject.height);
    }

    function updatePrimitives() {
        if (null !== startPoint){
            if ("filter" === startPoint.filter_type) {
                is_filter = true;
            }
        }

        var i;
        // remove old "lineSegmentRect_" primitives
        for (i = 0; i < diagramLine.children.length; ++i) {
            console.debug(diagramLine.children[i].objectName);
            if (diagramLine.children[i].objectName.indexOf("lineSegmentRect_") !== -1) {
                diagramLine.children[i].lineArea.entered.disconnect(lineEntered);
                diagramLine.children[i].lineArea.exited.disconnect(lineExited);
                diagramLine.children[i].destroy();
            }
        }

        for (i=0; i < middlePoints.length; ++i) {
            console.debug("***** middlePoints", middlePoints[i].x, middlePoints[i].y);
        }

        // create new primitives
        createPrimitive(endpoint1, middlePoints[0]);
        console.debug("***** point segment 0 start:", endpoint1.x, endpoint1.y);
        console.debug("***** point segment 0 end:", middlePoints[0].x, middlePoints[0].y);
        for (i = 0; i < middlePoints.length - 1; ++i) {
            createPrimitive(middlePoints[i], middlePoints[i+1]);
            console.debug("***** point segment ", i+1," start:", middlePoints[i].x, middlePoints[i].y);
            console.debug("***** point segment ", i+1," end:", middlePoints[i+1].x, middlePoints[i+1].y);
        }
        console.debug("***** point segment last start:", middlePoints[middlePoints.length - 1].x, middlePoints[middlePoints.length - 1].y);
        console.debug("***** point segment last end:", endpoint2.x, endpoint2.y);
        createPrimitive(middlePoints[middlePoints.length - 1], endpoint2);
    }

    onEndpoint2Changed: lineArrow.requestPaint()
    onFillStyleChanged: lineArrow.requestPaint()
    onLineWidthChanged: lineArrow.requestPaint()
    onVisibleChanged:   lineArrow.requestPaint()

    Component.onCompleted: {
        diagramField.selectionChanged.connect(diagramLine.selectionChanged)
        diagramLine.lineIsAboutToDestroy.connect(diagramField.lineIsAboutToDestroy)
        console.debug(objectName, "completed")
    }
    Component.onDestruction: {
        diagramField.selectionChanged.disconnect(diagramLine.selectionChanged)
        diagramLine.lineIsAboutToDestroy.disconnect(diagramField.lineIsAboutToDestroy)
        console.debug(objectName, "destructed")
    }

    Canvas {
           id: lineArrow
           objectName: "lineArrow"

           x: endpoint2.x - 8
           y: endpoint2.y - 13
           width: 16
           height: 16

           onPaint: {
               if (visible === false)
                   return;

               var ctx = getContext('2d');
               ctx.save();

               if (is_filter) {

                   x = endpoint2.x - 13;
                   y = endpoint2.y - 8;
                   // clear old boder of the arrow in selected state
                   ctx.strokeStyle = colorDiagramField;
                   ctx.lineWidth = selectedThickness+16;

                   ctx.beginPath();
                   ctx.moveTo(8, 0);
                   ctx.lineTo(8, 16);
                   ctx.stroke();

                   // redraw arrow
                   ctx.strokeStyle = strokeStyle;
                   ctx.fillStyle = fillStyle;
                   ctx.lineWidth = lineWidth;

                   ctx.beginPath();
                   ctx.moveTo( 3,  3);
                   ctx.lineTo( 3, 13);
                   ctx.lineTo(13,  8);
                   ctx.lineTo( 3,  3);
                   ctx.stroke();
                   ctx.fill();

                   ctx.restore();

               } else {
                   // clear old boder of the arrow in selected state
                   ctx.strokeStyle = colorDiagramField;
                   ctx.lineWidth = selectedThickness+2;
                   ctx.beginPath();
                   ctx.moveTo( 8, 13);
                   ctx.lineTo( 3,  3);
                   ctx.lineTo(13,  3);
                   ctx.lineTo( 8, 13);
                   ctx.stroke();

                   // redraw arrow
                   ctx.strokeStyle = strokeStyle;
                   ctx.fillStyle = fillStyle;
                   ctx.lineWidth = lineWidth;

                   ctx.beginPath();
                   ctx.moveTo( 8, 13);
                   ctx.lineTo( 3,  3);
                   ctx.lineTo(13,  3);
                   ctx.lineTo( 8, 13);
                   ctx.stroke();
                   ctx.fill();

                   ctx.restore();
               }
           }
       }


    states: [
        State {
            name: "selected"
            PropertyChanges { target: diagramLine; fillStyle: colorSelected }
            PropertyChanges { target: diagramLine; lineWidth: selectedThickness }
        },
        State {
            name: "hovered"
            PropertyChanges { target: diagramLine; fillStyle: colorLineHovered }
            PropertyChanges { target: diagramLine; lineWidth: hoveredThickness }
        }
    ]
}
