import QtQuick 2.0
import QtQuick.Layouts 1.1
import QtQuick.Controls 1.2
import QtQuick.Controls.Styles 1.2

Rectangle {
    id: debugVarsPanel
    objectName: "debugVarsPanel"

    property string title: qsTr("debugVarsPanel")
    property var model: varsModel

    signal varNameChanged(string oldName, string newName)

    function acceptChanges(row, role, oldValue, newValue) {
        console.debug("editor.onAccepted:",
                      "row=", row,
                      "role=", role,
                      ":", oldValue, "->", newValue)
        if (role === "name")
            varNameChanged(oldValue, newValue)
        model.set(row, role, newValue)
    }

    height: panelTitleText.height
    z: zVarPanel
    color: "#e0e0e0"
    border.color: "gray"
    opacity: 0.7

    Component.onCompleted: {
        varNameChanged.connect(diagramContainer.diagramField.varNameChanged)
    }

    ColumnLayout {
        anchors.fill: parent

        Text {
            id: panelTitleText
            text: title
            Layout.fillWidth: true
            height: paintedHeight + 2

            MouseArea {
                anchors.fill: parent
                onClicked: {
                    debugVarsPanel.height = (debugVarsPanel.height === height)
                        ? Math.round(150 * kDPI)
                        : height
                }
            }
        }

        TableView {
            id: varTableView
            objectName: "varTableView"

            Layout.fillWidth: true
            Layout.fillHeight: true
            model: debugVarsPanel.model
            alternatingRowColors: false
            frameVisible: false
            backgroundVisible: false

            Component {
                id: categoryDelegate
                Item {
                    Text {
                        anchors.fill: parent
                        anchors.leftMargin: 2
                        elide: styleData.elideMode
                        text: styleData.value
                        font.pixelSize: 12
                    }
                }
            }
            Component {
                id: varDelegate
                Item {
                    Text {
                        anchors.fill: parent
                        anchors.leftMargin: 2
                        elide: styleData.elideMode
                        text: styleData.value
                        font.pixelSize: 12
                        visible: !styleData.selected
                    }
                    Loader {    // Initialize text editor lazily to improve performance
                        id: loaderEditor
                        anchors.fill: parent
                        anchors.leftMargin: 2

                        Connections {
                            target: loaderEditor.item
                            onAccepted: {
                                acceptChanges(styleData.row, styleData.role,
                                              styleData.value, loaderEditor.item.text)
                            }
                        }

                        sourceComponent: styleData.selected ? editor : null
                        Component {
                            id: editor
                            TextInput {
                                id: textInput

                                anchors.fill: parent
                                text: styleData.value
                                clip: true
                                font.pixelSize: 12

                                Keys.onEscapePressed: { // reject editing
                                    text = styleData.value
                                    focus = false
                                }
                            }
                        }
                    }
                }
            }

            Component {
                id: typeDelegate
                Item {
                    ComboBox {
                        id: typeComboBox
                        anchors.fill: parent
                        anchors.leftMargin: 2
                        model: ["int", "string", "script"]

                        style: ComboBoxStyle {
                            background: Rectangle {
                                anchors.fill: parent
                                color: "transparent"
                            }

                            label: Text {
                                anchors.fill: parent
                                anchors.bottomMargin: 3
                                verticalAlignment: Text.AlignVCenter
                                elide: Text.ElideNone
                                text: typeComboBox.currentText
                            }
                        }

                        Component.onCompleted: currentIndex = find(styleData.value)
                        onActivated: {
                            debugVarsPanel.model.set(styleData.row, styleData.role, textAt(index))
                        }
                    }
                    Rectangle {
                        id: deleteButton
                        width: 16
                        height: 16
                        radius: 6
                        anchors.right: parent.right
                        color: "red"
                        visible: styleData.selected
                        border { color: "black"; width: 1 }

                        MouseArea {
                            anchors.fill: parent

                            onClicked: {
                                var varName = debugVarsPanel.model.get(styleData.row, "name")
                                console.debug("deleteButotn.onClicked:", styleData.row, varName)
                                varNameChanged(varName, "")
                                debugVarsPanel.model.removeRows(styleData.row)
                            }
                        }
                    }
                }
            }

            style: TableViewStyle {
                scrollBarBackground: Rectangle {
                    radius: 3
                    implicitWidth: 6
                    color: Qt.darker(debugVarsPanel.color, 1.1)
                }
                handle: Rectangle {
                    radius: 3
                    implicitWidth: 6
                    color: Qt.darker(debugVarsPanel.color, 1.4)
                }
                decrementControl: Item {}
                incrementControl: Item {}
            }

            rowDelegate: Rectangle {
                color: styleData.selected ? Qt.lighter(debugVarsPanel.color, 1.3) : "transparent"
            }

            TableViewColumn {
                id: categoryColumn
                role: "category"
                title: qsTr("Category")
                width: 50

                delegate: categoryDelegate
            }
            TableViewColumn {
                role: "name"
                title: qsTr("Name")
                width: (varTableView.width - categoryColumn.width - typeColumn.width) / 3

                delegate: varDelegate
            }
            TableViewColumn {
                role: "varName"
                title: qsTr("VarName")
                width: (varTableView.width - categoryColumn.width - typeColumn.width) / 3

                delegate: varDelegate
            }
            TableViewColumn {
                role: "value"
                title: qsTr("Value")
                width: (varTableView.width - categoryColumn.width - typeColumn.width) / 3

                delegate: varDelegate
            }
            TableViewColumn {
                id: typeColumn
                role: "type"
                title: qsTr("Type")
                width: 36 + 16  // combobox + delete button

                delegate: typeDelegate
            }

            onClicked: console.debug(debugVarsPanel.objectName, "onClicked:", row)
            onDoubleClicked: console.debug(debugVarsPanel.objectName, "onDoubleclicked:", row)
        }
    }
}
